/*
 *  hp82240.c
 *  
 *
 *  Created by Bernhard Emese on 18.04.15.
 *  Copyright 2015 Panamatik. All rights reserved.
 *
 */
#include <stdint.h>
#include "xc.h"

#define IRCLKPERIOD (16000000/4/32768)  // 32768 Hz = 30 us Period

#define MAXDIGITS 12
#define WSIZE 14

typedef uint8_t reg_t [WSIZE];

extern reg_t act_a;
extern reg_t act_b;

#define LF 0x0a

static void InitHP82240()
{
// use CCP2 PWM for 32768 Hz IR pulses
// PWM2 Output Low Level Idle Mode

	PR2=IRCLKPERIOD-1; // PWM Period 32,768 kHz
                 
// Set TMR2 Prescale Value 1:1 PostScale 1:1 Timer2 on

	TMR2=0;
	T2CON=0x00;   // Timer2, Prescale 1:1 , Postscale 1:1, Timer disabled
	CCP2CON=0x0c; // set PWM mode, reset lower 2 bits of PWM duty cycle
	CCPR2L=0;     // PWM DutyCycle 0%
}

static void ExitHP82240()
{
}

// send 1/2 bit, 14 cycles of 30 us,
// send burst of 8 pulses of 15 us each if Bit is set
static void SendHP82240Burst(uint8_t Bit)
{
	if(Bit & 1)
	  CCPR2L=IRCLKPERIOD/2; // set duty cycle 50% if high bit, otherwise is 0%

    TMR2IF=0;  // reset Timer2 to PR2 interrupt flag

	for(uint8_t i=0;i<14;i++) // half bit time is 14 cycles
	{
		while(TMR2IF==0) // wait 30 us until PWM cycle is finished
		  ;
		TMR2IF=0;

		if(i==7)    // stop pulse output after 8 pulses
		  CCPR2L=0; // set duty cycle 0% low level
  }
}

static void SendHP82240Bit(uint8_t Bit)
{
// one bit is high burst followed by low burst, else exchanged
   SendHP82240Burst(Bit);	
   SendHP82240Burst(Bit^1);	
}

// table for fast calculation of error correction bits
static const uint8_t correctiontab[8] = { 0x03, 0x05, 0x06, 0x09, 0x0a, 0x0c, 0x0e, 0x07};
 
// send one frame which is one ASCII character with start and error correction bits
// a frame is 30 cycles, 
static void SendHP82240Frame(uint8_t byte)
{
uint8_t byte1=byte;
uint8_t b=0;
uint8_t i=0;
uint16_t irbyte;

// calculate correction bits 9-11
	while(byte1)
	{
		if(byte1 & 1) b^=correctiontab[i];
		i++;
		byte1>>=1;	
	}

// equivalent enrolled calculation
//	if(byte1 & (1<<0) b^=0x03;
//  if(byte1 & (1<<1) b^=0x05;
//	if(byte1 & (1<<2) b^=0x06;
//	if(byte1 & (1<<3) b^=0x09;
//	if(byte1 & (1<<4) b^=0x0a;
//	if(byte1 & (1<<5) b^=0x0c;
//	if(byte1 & (1<<6) b^=0x0e;
//	if(byte1 & (1<<7) b^=0x07;

	irbyte=((uint16_t)b<<8) | byte;

	TMR2=0; // reset Timer 2 to zero
	TMR2ON=1; // start PWM Timer from 0
		
	for(i=0;i<3;i++)
		SendHP82240Burst(1); // send 3 start bits
			 
	for(i=0;i<12;i++) // send 12 bits, 4 bit correction, 8 bit data
	{
		SendHP82240Bit((irbyte & 0x800)!=0); // MSB first
		irbyte<<=1;
	}
		
	for(i=0;i<3;i++)
		SendHP82240Burst(0); // send 3 stop bits

	TMR2ON=0;
}

static const uint8_t AlphaDigits[6] = { 'r','F','o', 'P', 'E', ' '};

// send actually displayed X register
static void SendHP82240Digits()
{
	for(uint8_t n=0;n<MAXDIGITS;n++)
	{
		uint8_t a=act_a[WSIZE-1-n];
		uint8_t b=act_b[WSIZE-1-n];
		uint8_t digit=' ';
		if(b & 2)
			digit= a==9 ? '-' : ' ';  // sign ?
		else if(a>9) // alpha digits or blank
			digit=AlphaDigits[a-10];
		else
			digit=a + '0'; // numeric digit 0-9 
		if(digit!=' ' || !(b & 1)) // don't print space when only decimal point, can occur in program step
			SendHP82240Frame(digit);
		if(b & 1)  // decimal point ?
			SendHP82240Frame('.'); // add decimal point
	}
}

void SendHP82240LineFeed()
{
	InitHP82240();
	SendHP82240Frame(LF); // Line Feed
	ExitHP82240();
}

// send actual display register act_a, act_b, send 12 digits with comma, each needs 12,4 ms  = ca. 148 ms
void SendHP82240Display()
{
	InitHP82240();
	SendHP82240Digits(); // show actual Display
	SendHP82240Frame(LF); // Line Feed
	ExitHP82240();
}

// special characters
// divide = 129
// multiply = 130
// sqrt = 131
// Integral = 132
// Sigma = 133
// PI = 135
// <= 137
// >= 138
// != 139
// -> 141
// <- 142
// /x Mean ECMA 129
// Roll down ECMA 143
// Roll up ECMA 144

// use uint8_t opcodetab[] instead of uint8_t *opcodetab[] because it needs much less flash memory

// HP-25 program step opcodes

static const uint8_t opcodetab[] =
/* */ "GTO\0" "FIX\0" "SCI\0" "ENG\0" "STO\0" "RCL\0"
"\x8dHMS\0" "INT\0" "\x83\0" "Y^X\0" "SIN\0" "COS\0" "TAN\0" "LN\0" "LOG\0" "\x8dR\0"
"\x8dH\0" "FRAC\0" "x^2\0" "ABS\0" "SIN-1\0" "COS-1\0" "TAN-1\0" "e^x\0" "10^x\0" "\x8dP\0"
// numbers 0-9
"x<y\0" "x\x8ay\0" "x\x8by\0" "x=y\0" "LastX\0" "PSE\0" "?\0" "CLR\0" "REG\0" "STK\0" "\x1b\xf9\x81\x1b\xf8\0" "s\0" "\x85-\0" "?\0" "?\0"
"x<0\0" "x\x8a\x30\0" "x\x8b\x30\0" "x=0\0" "\x87\0" "NOP\0" "?\0" "DEG\0" "RAD\0" "GRD\0" "%\0" "1/x\0" "g \x85+\0" "M-\0" "M+\0"
"-\0" "+\0" "\x82\0" "\x81\0" ".\0" "R/S\0" "ENTER\0" "CHS\0" "EEX\0" "CLX\0" "X<>Y\0" "R\x1b\xf9\x8f\x1b\xf8\0" "\x85+\0" "M\x82\0" "M\x81\0"
;

static const uint8_t operatortab[] = { '-', '+', 0x82 , 0x81}; // use special characters for multiply divide

void SendHP82240mnemonic(uint8_t n)
{
	uint8_t *pt=(uint8_t *)opcodetab;
	uint8_t a;
	do
	{
		while(a=*pt++)
		{
			if(n==0)
  				SendHP82240Frame(a); // print mnemonic
		}
	} while(n--);
}

// n    Program step 0-49
// code Program code byte HP-25
void SendHP82240ProgramStep(uint8_t code,uint8_t mode)
{
	uint8_t a,l,h,digit;
	
	InitHP82240();

    if(mode & 1)
	{
		SendHP82240Digits(act_a); // show actual program step display  i.e. "01      13 00"
		SendHP82240Frame(' '); // space
	}

	h=code >> 4;  // high nibble
	l=code & 0x0f; // low nibble
	if(code<0x80) // handle GTO FIX, SCI, ENG, STO, RCL codes
	{
    	if(l<10)  // GTO, FIX, SCI, ENG
		{
			SendHP82240mnemonic(code<0x50 ? 0 : h-4);  // <0x50 GTO 50= FIX, 60=SCI 70=ENG
		    SendHP82240Frame(' '); // print space
			if(code<0x50)
				SendHP82240Frame(h | '0'); // print high digit of GTO instruction 0-4
			SendHP82240Frame(l | '0'); // print low digit 0-9
		}
		else
		{
			SendHP82240mnemonic(l==11 ? 5 : 4);  // STO, RCL, and STO arithmetics
			if(l>=12) // arithmetics ?
			  SendHP82240Frame(operatortab[l-12]); // add operator
			SendHP82240Frame(h | '0'); // register number 0-7
		}
	}
	else if(code>=0xa0) // 0x80- 0x9F are not defined
	{
		if(h==12)
			SendHP82240Frame(l | '0'); // numbers
		else
		{
			h=h<12 ? (h-10)*10+6 : (h-13)*15+26;
			SendHP82240mnemonic(h+l);
		}
	}
    SendHP82240Frame(LF);
	ExitHP82240();
}
