// newACT for HP Woodstock calculators
// 16 Mhz internal OSC 4 MHz Takt, 250 ns cycle
// 180 kHz ACT clock 5,5us Phi 1, Phi 2
// V1.00 10.8.2015 basic display and keyboard handling

#include <stdint.h>
#include "xc.h"

#pragma config WDTE=OFF // No Watchdog
#pragma config FOSC=INTOSC  // Internal Oscillator
#pragma config BOREN=OFF  // No Brown Out Reset
#pragma config MCLRE=ON  //
#pragma config CLKOUTEN=OFF
#pragma config CP=OFF // OFF
#pragma config IESO=OFF  // not used with INTOSC
#pragma config FCMEN=OFF  // not used with INTOSC
#pragma config PWRTE=OFF // Power On Timer enabled
#pragma config LVP=ON // Low Power programming

#define _XTAL_FREQ 16000000 // internal 16 MHz Oszillator, keyword for compiler

// Column 4-0 RA0,RA1,RA2,RA3,RA5 Analog Inputs
#define PRGMRUN	PORTAbits.RA4 // I Run Program Switch
#define TEST	LATAbits.LATA6 // O test strobe
#define VBATT   PORTBbits.RB0 // Digital Input 
#define SYNC	LATBbits.LATB3 // O Sync output
#define RCD		LATBbits.LATB4 // O Display Sync
#define LED		LATBbits.LATB5 // O LED (low active)
#define DATA	LATCbits.LATC0 // O RAM Data out
#define DATAIN	PORTCbits.RC0 // I RAM Data in
#define RESERV	LATCbits.LATC1 // I not used
#define PHI2	LATCbits.LATC2 // O Phi1 700 kHz PWM
#define PHI1	LATCbits.LATC3 // O Phi1 700 kHz PWM
#define ISAIN	PORTCbits.RC4 // IO Bus Data ROM Display
#define ISA		LATCbits.LATC5 // IO Bus Data ROM Display SDO connected to SDI

#define TRISPORTA 	0x3F // Analog Inputs PA0-PA3,PA5/ PA4 digital input, PA6,7 output
#define TRISPORTB 	0xc1 // all output, PB7,PB6 input
#define TRISPORTC 	0xD1 // all output except RX, SDI, DATA  // for ICD3 as Debugger

#define FALSE 0
#define TRUE  1

#define MAXDIGITS 12
#define COLUMNS 5
#define ROWS 7
#define WSIZE 14

#define ADLEVEL 175

// global variables
typedef uint8_t reg_t [WSIZE];

reg_t act_a;
reg_t act_b;
reg_t act_reg;

// local variabes

// flags
#define F_COLUMNCODE    (1<<0) // column code detected
#define F_DISPLAYON     (1<<1) // display on
#define F_SHOWINFO      (1<<3) 
#define F_PRGMRUN		(1<<4) // RUN/PRGM Switch, must be bit 4

static uint8_t flags;

// keyflags
#define KF_KEYPRESSED   (1<<0)
#define KF_KEYRELEASED  (1<<1)

static uint8_t keyflags;

static uint8_t Channel;  // actual AD Channel
static uint8_t column;
static uint8_t columncode;  // actual columncode
static uint8_t lastcolumncode;
static uint8_t keycode;

static uint8_t digit;       // display digit
static uint8_t displaydigit;// display digit number

static uint16_t romdata;	 // 10 bit opcode

static uint8_t TimerCnt;     // Counter increments every 8 ms
static uint8_t BoardType;    // hardware 0x21=HP21

void SendHP82240Display();

// will be called eeach 56-bit cycle (340 us) checks variable column, which is !=0 when key pressed
// calculate columncode, 4 bit row number and 4 bit column number
// row = 1-7, column = 1-5, columncode =  row<<4 | column;
void GetColumnCode()
{
	if(displaydigit>=1 && displaydigit<=ROWS) // row 1-7 is displaydigit 1-7
	{
		if(!(flags & F_COLUMNCODE) && column!=0) // key pressed in this row?
		{
			flags|=F_COLUMNCODE;  // accept only first columncode in lowest row to avoid false detecting in next row
			columncode=displaydigit<<4; // row 1-7
			if(BoardType!=0x21) // HP-25/22/27/29 hardware has different column layout than HP-21
			{
				if(column & 1)  // rotate right COLUMNS bits
					column|=1<<COLUMNS;
				column>>=1;
			}
     		while(column)
			{
				column>>=1;
				columncode++;
			}
		}
	}
}

// calculate keycode from columncode and set keypressed/keyreleased bits
static void readkeys()
{
	GetColumnCode();

	if(displaydigit==MAXDIGITS-1)  // after all rows are tested, set keycode, if button pressed then keycode!=0 
	{
		flags&=~F_COLUMNCODE;
		if(columncode==lastcolumncode) // columncode stable ?
		{
			if(columncode && keycode==0) // key pressed ?
			{
				keycode=columncode; // software dependent keycode
				keyflags|=KF_KEYPRESSED;  // pressed
			}
			else if(columncode==0 && keycode!=0) // key released ?
			{
				keyflags|=KF_KEYRELEASED; // released
			}
		}
		else
		{
			lastcolumncode=columncode;
		}
		columncode=0;  // will be set again in next keyboard cycle if any key is pressed
	}
}

// show ACT X register act_a act_b, set digit according to displaydigit
static void setdigit()
{
uint8_t a,b;

	digit=0x0f; // digit off
	if(flags & F_SHOWINFO)
		digit=act_reg[WSIZE-1-displaydigit];
	else if(flags & F_DISPLAYON) // calculator display on ?
	{
		a=act_a[WSIZE-1-displaydigit];
		b=act_b[WSIZE-1-displaydigit];
		digit=a | b<<4; // or decimal and sign
	}
}

// read ADValue Channel (0-4) and start conversion of Channel+1
void NextADCValue()
{
	if(ADRESH<ADLEVEL)
		column|=1<<Channel;
	ADCON0=0x01 | (Channel+1)<<2; // select next channel
	GO_nDONE=1; // start conversion next channel
}

// Phi1 Phi2 out relative to Data Out t=0
// OSC=180kHz*4 = 720kHz Cycle=138 ns  
static void PhiOut()
{
	NOP();NOP();NOP();
	PHI1=1;  // t=2778 (2800)ns d=694 (800) ns  
	NOP();NOP();NOP();
	PHI1=0; // t=3472 (3600) ns
	NOP();NOP();NOP();
	PHI2=1; // t=4167 (4400) ns
	NOP();NOP();NOP();
	PHI2=0;
}

// read opcode from ROM, output display digit
static void DataCycle(uint16_t addr)
{
	uint8_t i;

	setdigit();  // set display digit

	if(++displaydigit>=MAXDIGITS)
	{
		displaydigit=0;
		RCD=0; // display reset after 12 cycles
	}

// read keyboard 5 Columns analog values

	ADCON0=0x01; // select channel 0 for sampling before conversion start
	column=0;

	ISA=(digit & 0x01)!=0;	PhiOut(); // send LSB first
	ISA=(digit & 0x02)!=0;	PhiOut();
	ISA=(digit & 0x04)!=0;	PhiOut();
	ISA=(digit & 0x08)!=0;	PhiOut();
	ISA=(digit & 0x10)!=0;	PhiOut();
	ISA=(digit & 0x20)!=0;	PhiOut();
	ISA=(digit & 0x40)!=0;	PhiOut();
	ISA=(digit & 0x80)!=0;	PhiOut();
	GO_nDONE=1; // start AD conversion channel 0

	RCD=1; // display reset end

	for(i=0;i<8;i++)
	{
		ISA=0;	PhiOut();
	}
	Channel=0;
	NextADCValue();

	ISA=(addr & 0x01)!=0; 	PhiOut();
	ISA=(addr & 0x02)!=0;	PhiOut();
	ISA=(addr & 0x04)!=0;	PhiOut();
	ISA=(addr & 0x08)!=0;	PhiOut();
	ISA=(addr & 0x10)!=0;	PhiOut();
	ISA=(addr & 0x20)!=0;	PhiOut();
	ISA=(addr & 0x40)!=0;	PhiOut();
	ISA=(addr & 0x80)!=0;	PhiOut();

	Channel++;
	NextADCValue();

	ISA=(addr & 0x100)!=0;	PhiOut();
	ISA=(addr & 0x200)!=0;	PhiOut();
	ISA=(addr & 0x400)!=0;	PhiOut();
	ISA=(addr & 0x800)!=0;	PhiOut();

	Channel++;
	NextADCValue();

	for(i=0;i<8;i++) // 8 bit 0
	{
		ISA=0; PhiOut();
	}
	Channel++;
	NextADCValue();

	for(i=0;i<10;i++) // 10 bit 0
	{
		ISA=0;	PhiOut();
	}
	Channel++;
	NextADCValue();

	SYNC=1;

	TRISC|=0x20; // ISA input
	romdata=0;
	for(i=0;i<10;i++) // 10-bit instruction read
	{
		romdata>>=1; // read LSB first
		if(ISAIN)
			romdata|=0x200;
		PhiOut();
	}
	TRISC&=~0x20; // ISA output
	SYNC=0;
	
	readkeys();
}

void ClearInfo()
{
	for(uint8_t i=0;i<WSIZE;i++)
		act_reg[i]=0x0f; // all digits blank
	flags|=F_SHOWINFO;
}

// show n digits 32-bit hex number data at digit position
static void DisplayHex(uint32_t data,uint8_t digit,uint8_t digits)
{
	uint8_t a;
	
	for(uint8_t i=digits;i>0;i--)
	{
		a=data & 0x0f;
		if(a>=10)
		  a=a-10+0x10;  // show 0.1.2.3.4.5.
		act_reg[WSIZE-i-digit]=a;
		data>>=4;
	}
}

void CheckKeyPressed()
{
	if(keyflags & KF_KEYPRESSED) // key pressed ?
	{
		keyflags&=~KF_KEYPRESSED;
	}
	
	if(keyflags & KF_KEYRELEASED)  // key released ?
	{
		keyflags&=~KF_KEYRELEASED;
		keycode=0;  // key is processed

		if(flags & F_PRGMRUN) // print display if in RUN Mode
		{
			uint8_t i;
			for(i=0;i<WSIZE;i++)
				act_a[i]=act_reg[i];
			SendHP82240Display();
		}
	}
}

void CheckSlideSwitch()
{
	if((PORTA^flags) & F_PRGMRUN)  // RUN/PRGM Switch changed ?
	{
		flags&=~F_PRGMRUN;
		if(PORTA & F_PRGMRUN)
			flags|=F_PRGMRUN;
	}
}

// must be called at least every 8 ms
void DoTimer()
{

	if(TMR0IF) // every 8,1920 ms 16/4 Mhz/256/128
	{
		TMR0IF=0;

		TimerCnt++;

		CheckKeyPressed();
		CheckSlideSwitch();
	}
}

// Init special functions register at start up
static void Init()
{
	OSCCON=0x7A; // internal OSC 16 MHz

	PORTA=0x00;
	PORTB=0x20;   // LED off
	PORTC=0x00;
	TRISA=TRISPORTA;
	TRISB=TRISPORTB;
	TRISC=TRISPORTC;

	ADCON1=0x20; // 0x50=FOSC/16 1 MHz 0x20=FOSC/32 500kHz AD clock -> 24us (44 allowed)
	ANSELA=0x2F; // RA0-RA3 RA5 Analog Input, RA4 digital Input
	ANSELB=0x00; // RB0 Digital Input
	ANSELC=0x00; // all digital inputs

	OPTION_REG=0x06; // Timer0 internal clock, Prescaler assigned to Timer, rate 1:128, pull ups enabled
}

// main program entry and endless loop, ACT emulator
int main()
{
	Init(); // initialize special function registers
	
	while(1)
	{
		DoTimer();

		DataCycle(0); // run display and keyboard

		ClearInfo();
		DisplayHex(keycode,1,2);
		DisplayHex(flags & F_PRGMRUN,9,2);
	}
}
