// Multi calculator emulation for HP Classic Woodstock Spice calculators with PIC16F1519
// for HPLP Low Power Circuit hardware
// 16 Mhz Internal OSC 4 MHz clock, 250 ns cycle

// V1.00 25.06.2017 von HP25E 1.12 adaptiert, Umstellung auf Alphanumeric Display
// V1.00 18.10.2017 auf HPLP Hardware mit SST25 adaptiert
// V1.01 05.01.2018 first release version
// V1.02 keytables location changed, Reset selects HP-25, NROFCALCULATORS fixed
// V1.03 Welcome Text located in writable area, since V1.02 was accidentally write protected
// V1.04 12.03.2018 HP-34C steady display bug fixed, NMEA speed bug fixed, Fixed Constants 1.03 accidentally not written to Flash, HP-33 Spice bit was not set, HP-29 Beep g E+ in RUN mode
// V1.05 f STO GPS bug fixed, ON/OFF during GPS display fixed, SPICE repair kit, BROWNOUT ON prohibits erase of flash sector 0 at power up/down
// To do :

#include <stdbool.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include "xc.h"
#include "hp25lp.h"
#include "act.h"
#include "bcd.h"
#include "display.h"
#include "keyboard.h"
#include "hexentry.h"
#include "hp82240.h"
#include "memory.h"
#include "mnemonic.h"
#include "opcode.h"
#include "pcf2127.h"
#include "program.h"
#include "remote.h"
#include "serial.h"
#include "stopwatch.h"
#include "storage.h"
#include "trace.h"
#include "nmea.h"
#include "sst25PF040.h"
#include "undo.h"

#pragma config WDTE=OFF // No Watchdog
#pragma config FOSC=INTOSC  // Internal Oscillator
#pragma config BOREN=ON  // Brown Out Reset
#pragma config MCLRE=OFF  //
#pragma config CLKOUTEN=OFF
#pragma config CP=OFF // Code Protection OFF
#pragma config IESO=OFF  // not used with INTOSC
#pragma config FCMEN=OFF  // not used with INTOSC
#pragma config PWRTE=ON // Power On Timer enabled
#pragma config LVP=OFF // Low Voltage programming

// bit set is input
#define TRISPORTA 	0x18 // row select pins all output Prototype  K7 input, PRGM input, spare input
#define TRISPORTB 	0xFF // Col 1-5 input , ICSPCLK, ICSPDAT, /INT input
#define TRISPORTC 	0x91 // all output except RX SDO, CLKOUT
#define TRISPORTD 	0x00 // all output, 8 segments a-g d.p.
#define TRISPORTE 	0x00 // CE CE2 GPS output, RE3 MCLR input

// 7 segment display coding // c = 0x58  C = 0x39

#ifdef HP01
const char SegmentTab01[16+64+2]  = {
0x3f,0x06,0x5b,0x4f,0x66,0x6d,0x7d,0x07,0x7f,0x6f,0x80,0x40,0x80,0x5c,0xc0,0x00, // HP-01 characters '0-9' '.', '-', ':', 'o', '#', ' ' , semicolon is dot and minus together
};
#endif

#define MAXALPHA 102 // maximum alpha characters including some special characters

// 7 segment LED display code
const char SegmentTab[MAXALPHA-16]  = {
0x3f,0x06,0x5b,0x4f,0x66,0x6d,0x7d,0x07,0x7f,0x6f,0x50,0x71,0x5c,0x73,0x79,0x00, // HP-25 characters '0-9' 'r', 'F', 'o', 'P', 'E', ' ' ,
0x00,0x86,0x22,0xFF,0x6d,0xff,0xfe,0x02,0x39,0x0f,0x46,0x70,0x04,0x40,0x80,0x52, // Space ! " # $ % & ' ( ) * + , - . /  #=all segments
0x3f,0x06,0x5b,0x4f,0x66,0x6d,0x7d,0x07,0x7f,0x6f,0xc0,0x84,0x58,0x48,0x4c,0xd3, // 0-9 : ; < = > ?
0x7f,0x77,0x7c,0x39,0x5e,0x79,0x71,0x3d,0x76,0x06,0x1e,0x75,0x38,0x37,0x54,0x5c, // @ A-O  I same as 1
0x73,0x67,0x50,0x6d,0x78,0x3e,0x1c,0x7e,0x64,0x6e,0x5b,0x39,0x64,0x0f,0x01,0x08, // P-Z [ \ ] ^ _
0x2d,0x74,0x02,0x12,0x04,0x63 // <> h ' i , 
};

#ifdef SEGMENTTIMETABLE
// Display Timing. To get same brightness, digits with less segments have shorter display time
// coding two brightness in one byte makes program code shorter
const char SegmentTime[8+32+1] = {
0xf8,0xee,0xee,0xfe,0xff,0xef,0xef,0xf8, // HP-01 characters
0x88,0x8f,0xef,0xf8,0xEE,0xEE,0x88,0x8E, // Space, ! " # $ % & ' ( ) * + ' - . /
0xf8,0xee,0xee,0xfe,0xff,0x4c,0x88,0x8f, // 0-9 : ; < = > ?
0xff,0xfe,0xfe,0xef,0xe8,0x8f,0x8f,0xef, //@ A-O
0xdd,0x8e,0x8f,0xff,0xff,0xf8,0xe8,0x88, // P-Z [ \ ] ^ _
0xee };
#endif

#define ALPHAKEYS 8 // A,B,C,D,E,F,-,blank

// XYKEY, ROLLKEY, STOKEY,RCLKEY,SIGMAKEY,EEXKEY,CHSKEY,RSKEY
const uint8_t AlphaKey  [ALPHAKEYS] = { 0x21,0x22,0x23,0x24,0x25,0x34,0x33,0x74 };
const uint8_t AlphaDigit[ALPHAKEYS] = { 'A','B','C','D','E','F', '-',' ' }; // A B C D E F - blank

// global variables

// Flash Variables user area must be contigous from 0x4d0-0x4df
#define FLASHVARSIZE 16

uint8_t  FlashVars[FLASHVARSIZE] @ 0x4d0; // Flash Variables
uint16_t user_flags @ 0x4d0; // must be placed in user area
uint8_t  user_flagsl @ 0x4d0; // must be placed in user area
uint8_t  user_flagsh @ 0x4d1; // must be placed in user area
uint8_t  user_flags1 @ 0x4d2; // must be placed in user area 
uint8_t  DefaultTypeNr @ 0x4d3;
uint8_t  OperatingHours[4] @ 0x4d4; // operating time minutes hours 8 digit BCD
uint8_t  OperatingTime[2] @ 0x4d8; // operating time minutes hours 4 digit BCD
uint8_t  WriteCycles[3] @ 0x4da; // Flash write cycles 6 digit BCD 24-bit
uint8_t  TempOffset @ 0x4dd;
uint8_t  freserved1 @ 0x4de;
uint8_t  NrOfCalculators @ 0x4df;

uint16_t rombuf[32]; // buffer for flash write

uint8_t DisplayNr;
uint8_t MaxProgramSteps;
uint8_t act_weekday;
uint16_t Quotation;
uint16_t WaitKeyReleasedAddr; // firmware address for key released of calculators for SST Menmonic display

uint8_t flags;
uint8_t flags1;
uint8_t flags2;
uint8_t flags3;
uint8_t flags4;
uint8_t classicflags;
uint8_t systemflags;
uint8_t TypeFlags;
uint8_t RunBit;
uint8_t LongPressCnt;
uint16_t SWStartTime;
uint8_t KeyDownCnt;
uint8_t TraceCode;
uint16_t TimerCnt;
uint8_t SleepCnt;
uint8_t HPType;
uint8_t HPTypeNr;
uint8_t DispHPType;
uint8_t DispHPTypeNr;
uint8_t rightshift;
uint8_t keycode;
uint8_t DebugCommand;

volatile uint8_t decimal;
volatile uint8_t displaydigit;

static uint8_t EntryTimerCnt;
static uint8_t RepeatCnt;
static uint8_t SecondCnt;
static uint8_t OperatingSeconds;
static uint8_t RepeatTime; // must be power of 2
static uint8_t RepeatDelayTime;
static uint16_t TMR2Cnt;
static uint8_t BeepCnt;

uint16_t keyflags;
static uint16_t lastromaddr;

static uint8_t key,keydigit,sstkey,bstkey;
static uint8_t entrydigit;
static uint8_t PrimaryRegisters;

uint8_t StringIndex,ScrollIndex;
uint16_t PauseCnt;
uint8_t ReceiveTimeout;
uint8_t FlashPage;
uint16_t annuncmask;

#define USERSTRINGLEN 64

// Welcome Text
static uint8_t TickerCnt,TickerIndex;
static uint8_t TickerSpeed;

// Alarm Beeper
#define ALARMTIME 200  // 100 x Beep

static uint8_t AlarmCnt;
static uint8_t AlarmTimerCnt;

// Temperature and Voltage
uint16_t TempADCValue;
uint16_t RefADCValue;
//uint16_t TempSum; // Mittelwert fr Temperaturanzeige 16 Mittelwerte
uint8_t Temperature @ 0x525; // Temperature in Celsius
uint16_t Voltage @ 0x526; // Voltage in 1/100 V >0 damit vor erstem Analogwert keine LOWBAT angezeigt wird.

#ifdef BOOTLOADER

void interrupt intentry()  // will be located in boot loader area at 0x0004
{
#asm
	call 0x0102  ; lcall not necessary, ljmp doesnt work because routine must return to intentry in page 0
#endasm
}

// program entry for bootloader at absolute address 100h and 104h
// must be absolute addresses, because bootloader page is never changed
#asm
   	psect	boot_vec,global, abs, ovrld,reloc=256,class=CODE,merge=1,delta=2
	ORG 0100h                  ; reset vector
	ljmp __initialization
	ORG 0102h                  ; interrupt handler
	ljmp _IntHandler 	
  dw  VERSION                ; program version @ 0x104
	dw	REVISION
#endasm

#endif

// PortA selects digit row 1-12, if bit7=1  bit 3,4,5,6 are row1,2,3,12, bit7=0, bit 3,4,5,6, must be high
// low order RA4, RA5, RA6. bit3-0 1100, 1000, 1011, 1101,1001,1110,1010,1111
// HPLP 1.1 SEL ABC= RA0-2 SEL1 = 3,4,5,2,1,12,-,- SEL2 -,-,6,7,11,10,9,8 0x80=SEL1 0x40=SEL2
const uint8_t DigitMask[12] = { 0x84, 0x83, 0x80, 0x81, 0x82, 0x42, 0x43, 0x47,0x46,0x45,0x44,0x85, };

/*-----------------------------------------------------------------------------------------
 Interrupt handler, interrupt service routine
 Display digit Interrupt
 UART Receive Interrupt when byte received
 ------------------------------------------------------------------------------------------*/
#ifdef BOOTLOADER
void IntHandler () // @ 0x102 // address will be overwritten by "interrupt" specifier
#else
void interrupt IntHandler () // address will be 0004
#endif
{
  uint8_t i,digit;

	if(TMR2IF) // Timer 2 interrupt for display and keyboard scan, variable time about 200-400 us 
	{
    TMR2IF=0;
    flags1|=F_TICK;

// show multiplexed LED digit. This method does show segments with same brightness.

		if(++displaydigit>=MAXDIGITS)       //  check keyboard after each completed display cycle
   	{
      flags1|=F_DISPTICK;
			displaydigit=0;
      ReceiveTimeout++;   // Zhler fr ReceiveTimeout 250 = ca 800 ms

      GetColumnCode(); // Get row column code from keyrow and keycolumn

   		if(ColumnCode!=lastcolumncode)    // key detected ?
     	{
       	lastcolumncode=ColumnCode;
     		prellcnt=PRELLTIME;
     	}
   		if(prellcnt && --prellcnt==0)
   		{
        readkeys(); // wird aufgerufen wenn Columncode nach Entrprellung gewechselt hat
     	} 
      keyrow=0; // reset keyrow variable for next key scan
 		}

    digit=' ';
    if(!(flags3 & F_DISPLAYOFF))
       digit=act_dsp[WSIZE-1-displaydigit];

    i=digit & 0x7f; // clear bit 7 decimal point
    if(i>=32)  // ASCII alphanumeric ? small characters are not allowed
      i-=16;	// convert to SegmentTab index, SegmentTab is not ASCII indexed, saves flash space

#ifdef HP01
    if(HPType==0x01 && i<16)
      digit=SegmentTab01[i] | (digit & 0x80); // keep decimal point
    else
#endif
      digit=SegmentTab[i] | (digit & 0x80); // keep decimal point

// last digit is still on until now, dark time must be minimized to get best brightness per mA
// following instructions are optimized for minimum dark time between digit change
// dark time is only 4 instructions, 1 us @ FOSC 16 MHz

    LATD=0; // alle Segmente ausschalten vor digit Umschaltung
#ifdef SPICE
    SEMI=0;  // Semicolon ausschalten
#endif
  	LATA=DigitMask[displaydigit]; // select row R1-12 sequentially
   	LATD=digit;              // set segments, now visible
#ifdef SPICE
    if(Semicolon & (1<<displaydigit))
      SEMI=1;
#endif
		i=PORTB & 0x3E;          // read all 5 columns PB1-5

    if(i!=0x3E && keyrow==0)
    {
#ifdef SPICE
      if(displaydigit>=1 /*&& displaydigit<=7*/) // Spice hardware read keycode from digit 1-7
        keyrow=8-displaydigit; // row 1-7 a key is pressed in this row, row layout in reverse order compared to Woodstock
#else
      keyrow=displaydigit+1; // a key is pressed in this row
#endif
      keycolumn=i;
    }

    if(!(flags4 & F_BEEP))    // don't change PR2 during alarm beep to keep PWM frequency constant
    {
#ifdef SEGMENTTIMETABLE
// Helligkeitskorrektur mit Tabelle, jeder Tabelleneintrag enthlt die Helligkeit fr zwei Zeichen
      j=SegmentTime[i>>1];  // Dieser Code ist wegen der halb so grossen Tabelle krzer
      if(i & 1)
        PR2=(j & 0x0F) <<2; // Helligkeitsjustierung der einzelnen Zeichen, entspricht Zeiten von 200-400us
      else
     	  PR2=(j & 0xF0) >>2;
#else
// Helligkeitskorrektur in nur 3 Stufen ist ausreichend und spart eine komplette Helligkeitstabelle
// Der einzelne Dezimalpunkt nicht abgefragt und heller als normal angezeigt, kommt aber selten vor und fllt nicht auf.
      if(digit==0x06) // 1 und I haben nur 2 Segmente, deshalb nur kurz anzeigen
        PR2=0x18;
      else if((digit & 0x3f)==0x3f)  // mehr als 6 segmente, 0,8, lang anzeigen
        PR2=0x28;
      else
        PR2=0x20; // alle anderen Zeichen haben mittlere Helligkeit
#endif
    }
	}
#ifdef SERIALPORT
	if(RCIF) // serial interrupt ?
  {
		ReceiveSer(RCREG);
  }
#endif
}

void ClearInfo()
{
//  memset(act_reg,' ',WSIZE); wird nicht effektiv bersetzt
	for(uint8_t i=0;i<WSIZE;i++)
		act_reg[i]=' '; // all digits blank
	flags|=F_SHOWINFO;
}

// optimized,  does'nt clear TMR0, therefore uncertainty=8 ms
// Wait time will be 8,2*(n-1) +8,2/-0.0
void Wait8ms(uint8_t n)
{
  do
  {
    TMR0IF=0;
    while(TMR0IF==0); 
  }
  while(--n);
}

// one tick is 32 us, max 255 ticks = 8,16 ms
void DelayTicks(uint8_t ticks)
{
    uint8_t  t=TMR0;
		while((uint8_t)(TMR0-t)<ticks); // TIMER0 runs with 4000000/128 = 31250 Hz, /10 = about 3000 Instructions/sec
}

void StartDisplay()
{
  TMR2=0;
	T2CON = 0x07; // Timer 2 On , 5= Prescaler 1:4 6=Prescaler 1:16  7=Prescaler 1:64
  PR2=255; // 40 ms 244 Hz Interrupt, dynamic timing
	TMR2IE=1;
}

void StopDisplay()
{
  TMR2IE=0;
 	LATD=0;     // all segments off, because rows/digits 1-6 will be selected
	TMR2=0;
	T2CON=0x00; // Timer2, Prescale 1:1 , Postscale 1:1, Timer disabled
}

void IncrementWriteCycles()
{
  IncrementBCD(WriteCycles,3,0x99);
}

void Writerombuf(uint8_t page,uint16_t addr)
{
  ReadText(7); // Write Flash
  SST25WriteToPage(page,addr,rombuf8);
	IncrementWriteCycles();
	flags&=~F_SHOWINFO;
}

void Readrombuf(uint8_t page,uint16_t addr)
{
  SST25Read(page,addr,rombuf8,FPAGESIZE);
}

// Beep uses CCP1 PWM using TMR2, during beep the display is switched to fixed equal periods
void SoundBeep()
{
  flags4|=F_BEEP;
  PR2=66; // 80;      // 66=950 Hz, 80=800 Hz  @ 4 MHz  Prescaler 1:16  1000000/16/80=781,25 Hz, Higher Freq has better resonance
	CCPR1L=40;          // PWM DutyCycle 50%
	CCP1CON=0x0c; // setze PWM Mode, reset lower 2 bits of PWM Duty Cycle
}

void EndSoundBeep()
{
  flags4&=~F_BEEP;
	CCPR1L=0x00; // PWM DutyCycle 0 %
	CCP1CON=0x00; // kein PWM Mode
  BUZZER=0;     // set low because connected to base of NPN transistor
}

// Get stopwatch time as int in 1/100 s
uint32_t GetSWTime()
{
  uint32_t Time=0;
  uint32_t k = 1;
  for (uint8_t i = 0; i <8; i++) // sum 8 digits HH-MM-SS.TT
  {
    Time += act_sw[i]*k;
    if(i==3 || i==5)
      k *= 6;
    else
      k *= 10;
  }
  return Time;
}

// Set Timer1 Value and SWStartTime from Time in 1/100 s
// Timer1 can hold 512 seconds
// SWStartTime unit is 512 seconds
// if SWDEC is set, store negative Timer Value
// SWDEC will be set before calling op_atosw
// when act_a=0 is doesnt matter
void TMR1Reset(uint32_t Time)
{
  uint16_t t;

  SWStartTime=Time/51200L;  // Nr Of 512 seconds interval, max 703 for 100 hours
  t=Time-SWStartTime*51200L; // 1/100 seconds difference
  t=(uint32_t)t*32/25;  // convert 1/100 s to 128/s *128/100
  if(hp01flags & F_SWDEC)
  	t=-t;               // store negativ value for alarm on overflow when countdown reaches zero

  TMR1ON=0; // stop hardware timer before writing
  TMR1L=t;
  TMR1H=t>>8;
}

void ReadText(uint8_t n)
{
#ifndef XDEBUG  
  char buf[12];
  SST25Read(TEXTPAGE,FTEXTADDR+n*12,buf,12); // read 14 bytes Text to act_reg from Flash
  for(uint8_t i=0;i<MAXDIGITS;i++)
    act_reg[WSIZE-1-i]=buf[i];

#ifdef SPICE
  if(buf[0]!=' ') // wenn digit 1 kein Space ist, dann um ein digit nach rechts verschieben.
  {
    for(uint8_t i=0;i<MAXDIGITS;i++)
      act_reg[WSIZE-2-i]=act_reg[WSIZE-1-i];
    act_reg[WSIZE-1]=' '; // erstes digit lschen, sonst wird evtl. Minuszeichen angezeigt
  }
  
#endif
	flags|=F_SHOWINFO;
  DoDisplay();
#endif
}

void ShowText(char n)
{
  ReadText(n);
  flags3|=F_INFOTEXT; // Text wird quittiert mit nchstem Tastendruck
}

void RemoteDataReceived()
{
  TriggerSleepTime();
#if 0 // excluded to save program space
  if(InBuf[0]==1)
  {
    ReadText(54);  // COMMUNICATE
    keyflags|=KF_SHOWINFO;
    TimerCnt=0;
//    flags3|=F_INFOTEXT; // Text wird quittiert mit nchstem Tastendruck
  }
#endif
}

#ifdef RIGHTSHIFT
// calculate digit position for right justified display
void SetRightShift()
{
	rightshift=0;  // left justified
	if(user_flags & UF_RIGHTADJUST)
	{
		if(act_a[WSIZE-1-11] != 0x0f) // exponent an ? dann kein rightshift
			return;

		for(uint8_t i=0;i<11;i++)  // scan 10 digits for blank shift until number is right adjusted
		{
			if(act_a[WSIZE-2-i] == 0x0f)
			{
				rightshift=10-i;
				return;
			}
		}
	}
}
#endif

void ShowMantissa()
{
  ClearInfo();
  dest=act_reg;
  if(TypeFlags & TF_21_22_27_67) // HPType==0x21 || HPType==0x22 || HPType==0x27 || HPType==0x67)
  {
    src=act_c; 
    first=0; last=WSIZE-1;
    reg_copy();
  }
  else // HP-25/29
  {
    src=act_n; 
    if(HPType==0x29)
      src=act_m; 
    reg_decompress();
  }
	act_reg[2]=act_reg[WSIZE-1]=' ';
  flags3|=F_INFOTEXT;
}

static void EnterCalculator()
{
//  if(DisplayNr==DISPNMEA)
//   	keyflags|=KF_CLEARPREFIX|KF_ALLOWKEY;
  DisplayNr=DISPCALCULATOR;
  ClearInfo();
  flags&=~F_SHOWINFO; // lsche act_reg fr ShowMnemonic
  flags3&=~F_INFOTEXT;
#ifdef MNEMONICS
  if(!(act_flags & AF_RUNMODE))
    ShowMnemonic();
#endif
//	if(user_flags & UF_SYMBOLDISPLAY)
//		ShowPI();
}

// cl register has different format than dsp,  6 bytes HHMMSS always 24 hours
void timetodsp(uint8_t *time)
{
static const uint8_t tab1[6] = { 5,6,8,9,11,12 };

  for(uint8_t i=0;i<6;i++)    // spart 3 instructions
    act_reg[tab1[i]] = time[i];
#if 0
  act_reg[5] = time[0]; // copy HH:MM:SS
  act_reg[6] = time[1];
  act_reg[8] = time[2];
  act_reg[9] = time[3];
  act_reg[11] = time[4];
  act_reg[12] = time[5];
#endif
  act_reg[7] = '.';
  act_reg[10] = '.';
  if(user_flags & UF_TIMEAMPM) // 12 Stunden format
  {
    uint8_t h=time[4] + time[5]*10; // 0-23
    act_reg[3] = h<=12 ? 'A' : 'P';  // AM or PM
    act_reg[2] = 'M';
    if(h>=13)
    {
      h-=12;
      act_reg[11] = h %10;
      act_reg[12] = h/10;
    }
  }
}

static void ShowAlarmTime()
{
  timetodsp(act_al);
  act_reg[7] = ':';
  act_reg[10] = ':';
  if(hp01flags & F_ALARMACTIV)
     act_reg[4]='-';  // Alarm Minus
  if(flags4 & F_ALARMDAILY)
     act_reg[4]|=0x80;  // show daily alarm with dot
}

void ShowDate()
{
  act_reg[3] = act_cl[6]; // copy DD:MM:YY
  act_reg[4] = act_cl[7];
  act_reg[5] = '0';
  act_reg[6] = '2';
  act_reg[7] = '-';
  act_reg[10]= '-';
  if(user_flags & UF_DATEUS) // Amerikanisches Anzeigeformat MM-TT-YY
  {
    act_reg[8] = act_cl[10];
    act_reg[9] = act_cl[11];
    act_reg[11] = act_cl[8];
    act_reg[12] = act_cl[9];
  } else
  {
    act_reg[8] = act_cl[8];
    act_reg[9] = act_cl[9];
    act_reg[11] = act_cl[10];
    act_reg[12] = act_cl[11];
  }
}

// Stopwatch register is 8 digits HH:MM:SS.CC
void ShowStopwatch()
{
  uint8_t i,j,k;

  k=8-1;
  if((act_sw[6] | act_sw[7])==0) // wenn hours=0 nur 6 digits anzeigen
  {
     k=6-1;
     act_reg[11]=' ';
     act_reg[12]=' ';
     act_reg[13]=' ';
  }
  for(j=3,i=0;;i++,j++) // beginne mit zweiten digit von rechts mit 1/100
  {
     act_reg[j]=act_sw[i]; // optional use all 12 digits in HP-25 HH:MM:SS, original only HH:MM:SS can be shown
     if(i==k)
       break;
     if(i<=5 && i & 1)
       act_reg[++j]='-';  // skip display digit every two digits
  }
	act_reg[5]='.'; // decimal dot
}

uint8_t GetTickerText(uint8_t index)
{
uint8_t a;
#if 0
uint8_t page;
uint16_t addr;

  page=QUOTATIONPAGE;
  addr=QUOTATIONADDR;
  if(Quotation==0)
  {
    page=TICKERTEXTPAGE;   // located in writable area
    addr=TICKERTEXTADDR;
  }
  SST25Read(page,addr+Quotation+index,&a,1);
#endif

  if(Quotation==0)
    SST25Read(TICKERTEXTPAGE,TICKERTEXTADDR+index,&a,1);
  else
    SST25Read(QUOTATIONPAGE,QUOTATIONADDR+Quotation+index,&a,1);

  return a; 
}

// Show EOT terminated ticker text
void ShowTicker()
{
uint8_t i,j,a;

  flags|=F_SHOWINFO;
  for(j=TickerIndex,i=0;i<MAXDIGITS;i++)
  {
    a=GetTickerText(j++);
    if(a==TICKEREOT)  // end of text # ?
    {
   		a=' ';  // show EOT marker as space
     	j=0;    // start ticker text from begin
    }
    act_reg[WSIZE-1-i]=a;
  }
#ifdef SPICE
  act_reg[WSIZE-1]=' ';  // digit 1 darf nicht benutzt werden, da nur Minuszeichen angezeigt werden kann
#endif
}

#ifdef HPSERIALNUMBER
void ShowSerialNumber()
{
  Readrombuf(FLASHVARPAGE,FLASHVARADDR);  // Read serial No from Flash to rombuf
  act_reg[WSIZE-1]= 'S'; //Sr.
  act_reg[WSIZE-2]= 'R'|0x80;
	for(uint8_t i=0;i<10;i++)
 	 act_reg[WSIZE-3-i]=rombuf8[HPSERIALNRADDR+i];
}
#endif

static void ShowRevision()
{
uint8_t i;

  ReadText(30);  // R       SR
  act_reg[WSIZE-2]= VERSION|0x80;
  act_reg[WSIZE-3]= REVISION>>4;
  act_reg[WSIZE-4]= REVISION&0x0F;
//  act_reg[WSIZE-7]='R'|0x80;
  Readrombuf(FLASHVARPAGE,FLASHVARADDR);  // Read serial No from Flash
#ifdef SPICE
  for(i=0;i<4;i++)  // seriennummer 4-stellig anzeigen, da nur 11 digits vorhanden sind
		act_reg[(WSIZE-8)-i]=rombuf8[SERIALNRADDR+i+1];
#else
  for(i=0;i<5;i++)  // seriennummer 5-stellig anzeigen
		act_reg[(WSIZE-8)-i]=rombuf8[SERIALNRADDR+i];
#endif

#ifdef HPSERIALNR
	if(rombuf8[HPSERIALNRADDR]!=0) // calculator case serial number present ?
    flags3|=F_SHOWSERIAL;
#endif
  flags3|=F_INFOTEXT;
}

// Show Logo HP-21/HP-25
void ShowLogo()
{
  ReadText(29); // HP-00E LP
	act_reg[WSIZE-1-5]=DispHPType >> 4;
	act_reg[WSIZE-1-6]=DispHPType & 0x0f; // HP-21,22,25 etc.

	if(user_flags & UF_HP82240)
	{
		act_reg[WSIZE-1- 9]='I'; // Ir  Infrared
		act_reg[WSIZE-1-10]='R';
	} else
#ifdef NMEAMENU
  if(ENABLEGPS)
  {
    act_reg[WSIZE-1- 9]='G';  // Global Positioning System
    act_reg[WSIZE-1-10]='P';
    act_reg[WSIZE-1-11]='S';
  } else
#endif
  { 
    act_reg[WSIZE-1- 9]='L'; // Low Power
    act_reg[WSIZE-1-10]='P';
  }
}

void ShowLogoInfoText()
{
  if(user_flags & UF_SHOWLOGO)
  {
    ShowLogo();
    flags3|=F_INFOTEXT;
  }
}

void ShowVoltage()
{
#ifdef SHOWVOLTAGE

uint8_t n,t;

  ReadText(11); // "VOLT",7);
  n=Voltage/100;
  t=Voltage-n*100;
  act_reg[WSIZE-1-3]=n|0x80;  // show 3 digits Voltage 2.00 - 3.30 Volt
  act_reg[WSIZE-1-4]=t/10;
  act_reg[WSIZE-1-5]=t%10;
#endif
}


// Show Temperature as C or F
// 1 digit ist ca 0.9 C. Der AD Wandler springt meist um 2 digits, sodass die Anzeige um 2 Grad ungenau ist.
// Der Slope ist fest auf 4x0,132 = 0,528 Volt/100C eingestellt siehe Microchip AN1333
// vom gemessenen TempADCValue wird 2V = ca. 400 abgezogen, da im Temperaturbereich der Wert immer ber 2V liegt
// und damit ein 16-bit berlauf verhindert wird
void ShowTemperature()
{
#ifdef SHOWTEMP
// TempADCValue ergibt bei ca 20C folgende Werte 0xF4 bei 3.0V 0x9d bei 2,7V 0x55 bei 2,5V
// RefADCValue 2.048V ist etwa 2/3 von 10-bit bei Vdd 3.0V

// Vt = 0.659  Temperature C + 40 * (0.00132)

// Messung mit Reference Voltage=2.048V und Temperature sensor High Range TSRNG=1 Vout=Vdd-4*4Vt
// gemessen bei 3.0V   RefADCValue 0x2be(702)(2.99V) TempADCValue 0xF4 Diff 0x1ca
// gemessen bei 2.75V  RefADCValue 0x2fa(762)(2.75V) TempADCValue 0xAA Diff 0x250
// gemessen bei 2.5V   RefADCValue 0x347(839)(2.50V) TempADCValue 0x57 Diff 0x2f0
// Messung mit Reference Voltage=2.048V und Temperature sensor High Range TSRNG=0 Vout=Vdd-2*4Vt  Vdd=always 1023
// gemessen bei 3.0V   RefADCValue 0x2be(2.99V) TempADCValue 0x284(644) 1.88V  2.99-1.88=1.11V Vt Temperaturwert
// gemessen bei 2.75V  RefADCValue 0x300(2.73V) TempADCValue 0x25F(607) 1.61V  2.73-1.61=1.12V
// gemessen bei 2.5V   RefADCValue 0x34A(2.49V) TempADCValue 0x236(566) 1,37V  2.49-1.37=1.12V

// Umrechnung mV in digits  digits=mV*RefADCValue/2048
// Umrechnung digits in mV  mV=digits*2048/RefADCValue

// uint16_t v=(uint32_t)205*1024/RefADCValue; // Versorgungsspannung in 1/100 Volt 205=gerundet 2048/10
// Slope ist -0,165V/125C pro Diode laut Diagramm in AN1333  *2=0,33V  (*4=0.66V) oder 2(4)*0,00132V pro C
// Fr uns relevant ist der feste Wert 0,330/125C oder 0,264V/100C, 0,338/128C
// uint16_t Slope=(uint32_t)RefADCValue*264/2048; // Slope ist 2(4)*0.132V/100C = 0,264V/100ergibt ca. 200 bei 2.7V, 216 bei 2.5V, 180 bei 3.0V
// der Digitwert von Slope ist abhngig von Vdd, er wird kleiner bei grerer Versorgungsspannung
// daher auf Vdd normalisieren
// Berechne Vt in Einheiten von 1/100V;
// Vt liegt im Bereich von 3.0-1.3V(-40) 1.14V(0) bis 3.0-1.0V (+85) // d.h. ca vt 256-512
// Vt liegt im Bereich von 2.5-1.3V(-40) bis 2.5-1.0V (+85)

    uint8_t t;

    t=Temperature; // C

    if(flags3 & F_FAHRENHEIT)
      t=(uint16_t)t*9/5+32;

    uint8_t n=t/100;
    t-=n*100;
    if(n==0)
      n=0x0f; // blank
    act_reg[WSIZE-1-6]=n;  // show up to 3 digits temperature 0-85C / 0-185F
    act_reg[WSIZE-1-7]=t/10;  //
	  act_reg[WSIZE-1-8]=t%10;  //
	  act_reg[WSIZE-1-9]=0x65;  // displays upper o degree character
// show C or F
    if(!(flags3 & F_FAHRENHEIT))
  	  act_reg[WSIZE-1-10]='C';  // HP-29 shows C
    else
  	  act_reg[WSIZE-1-10]='F';  // HP-25 shows F

#else
  ReadText(12);  // show fixed temp
#endif
}

#ifdef SHOWROM
// Show ROM address and opcode either 12-bit octal or hex
static void ShowROM()
{
  uint16_t romdata;

	TimerCnt=0; // prevent repeating on entry
	
  SST25Read(FlashPage,lastromaddr,(uint8_t *)&romdata,2);
//	romdata=ReadInternalFlash(lastromaddr); // read from internal ROM
	if(flags1& F_SHOWHEX)
	{
		DisplayHex(lastromaddr,2,4);
		DisplayHex(romdata,7,4);
	} else
	{
		DisplayOct(lastromaddr,1,5);
		DisplayOct(romdata,7,4);
	}
	act_reg[WSIZE-1-0]='A';  // A for address
	act_reg[WSIZE-1-1]=FlashPage;
	act_reg[WSIZE-1-11]=flags1 & F_SHOWHEX ? 'H' : 'O';  // H or o
}

// verarbeitet keys von ShowROM
void DoShowROM()
{
	if(keyindex==PLUSKEY || hpcode==0x11)
		lastromaddr=lastromaddr+1;
	else
	  lastromaddr=lastromaddr-1;
}

#endif


// show and toggle user_flags
static void ShowUserFlags()
{
uint8_t i,digit;

	act_reg[WSIZE-1-1]='F'; // show F for flags

  for(i=0;i<10;i++)
	{
      digit=0x0f;
      if(i<6)
      {
        if(user_flagsh & 4<<i)  // - anzeigen wenn Flag 10-15 gesetzt
          digit='-';
      } else
      {
        if(user_flags1 & 1<<(i-6)) // - anzeigen wenn Flag 16-19 gesetzt
          digit='-';
      }
			if(user_flags & 1<<i)  // . anzeigen wenn Flag 0-9 gesetzt
        digit|=0x80;
			act_reg[WSIZE-3-i]=digit;
	}
}

// display all segment
static void ShowLEDTest()
{
	for(uint8_t i=0;i<WSIZE;i++)
		act_reg[WSIZE-1-i]='8' | 0x80;  // 8.
}

#ifdef PRGMCHECKSUM
// berechne die Checksumme des aktuell geladenen Programms im RAM
uint16_t GetPRGMChecksum()
{
uint16_t chksum=0;
uint8_t i,j;

  GetProgramAddr(0); // setze ProgramRegs und ProgramOffset abhngig von HPType

  for(i=0;i<ProgramRegs;i++) // loop all registers of program
	{
    getrampt(ProgramOffset+i);
		for(j=0;j<7;j++)
			chksum+=dest[j];
	}
	return chksum;
}

uint8_t GetMaxPrograms()
{
  uint8_t n=MAXPROGRAMS+USERPROGRAMS;
  if(HPType==0x25)
    n+=HP25APPLICATIONPROGRAMS; // HP-25 has additional more programs
  if(HPType==0x29)
    n+=HP29APPLICATIONPROGRAMS; // HP-25 has additional more programs
  if(HPType==0x67)
    n+=STANDARDPACPROGRAMS; // 15 StandardPac Programs + 3 additional
  return n;
}

// sucht die Nummer des Programms aus der Library
// mode = F_COMPAREPRGM oder F_ISEMPTYPRGM
// return Nr des Programms oder -1
uint8_t GetProgramNr(uint8_t mode)
{
uint8_t i,n;

  n=GetMaxPrograms();

	flags2|=mode;
	i=MAXPROGRAMS;  // suche zuerst in den Speicherpltzen 0-9 (100-109)
	for(;;)
	{
		if(ReadProgram(i))  // freies oder identisches programm gefunden ?
			break;
    i++;
		if(i==MAXPROGRAMS) // keinen freien Patz gefunden in 100 library programmen
		{
	  	i=255;
			break;
		}
  	if(i==n) // keinen freien Platz in den user programmen
	  	i=0; // suche weiter in 00-99 fr compare auch in fixed programs suchen
	}
	flags2&=~mode;
	return i;
}

#if 0
// zeigt die Program Checksumme CheckSum an mit Programmnummer i
// wenn i==255 wird keine Programmnummer angezeigt.
void ShowCheckSum(uint8_t i)
{
}
#endif

// show program checksum
void ShowPRGMChecksum()
{
uint16_t chksum;
uint8_t i;

	chksum=GetPRGMChecksum();
	i=GetProgramNr(F_COMPAREPRGM);

  ReadText(31); // Pr.

	if((HPType==0x34 && chksum==0300) || chksum==0 )
	{
		act_reg[WSIZE-1-3]='-';
		act_reg[WSIZE-1-4]='-';
	}
  else if(i>=MAXPROGRAMS && i<MAXPROGRAMS+USERPROGRAMS)
	{
		act_reg[WSIZE-1-5]=i-MAXPROGRAMS;  // one digit
    ProgramNr=i;
	}
	else if(i!=255)
	{
//		act_reg[WSIZE-1-4]=i/10;  //  at least two digits
//		act_reg[WSIZE-1-5]=i%10;  //
    DisplayDecimal(i,3);
    ProgramNr=i;
	}
	DisplayOct(chksum & 0xfff,7,4); // right justified
  flags3|=F_INFOTEXT;
}
#endif

// read constants pi and e from flash memory
static const uint8_t pi[] = {  0x00,0x40,0x65,0x92,0x15,0x14,0x03 } ;
static const uint8_t  e[] = {  0x00,0x80,0x82,0x81,0x82,0x71,0x02 };

// zeigt die Zahl PI oder E im display an
void ShowPI()
{
	uint8_t i,n,k;

//  SST25Read(FCONSTANTPAGE,FCONSTADDR+100*8,rombuf8,16); // read 2 constants pi, e , needs more program space
  
    uint8_t *pt=(uint8_t *)act_c;

	  k=3;
	  for(i=0;i<WSIZE/2 && k;i++)  // show PI or E
	  {
		  n=pt[2*i+1]<<4 | pt[2*i];  // 	n=act_c[2*i+1]<<4 | act_c[2*i]; causes fixup overflow
		  if(n!=pi[i] /*rombuf8[i]*/)
			  k&=~1;
		  if(n!=e[i] /*rombuf8[i+8]*/)
			  k&=~2;
	  }
	  if(k && !(flags3 & F_INFOTEXT))
	  {
  		flags1|=F_SHOWPI;
	  	if(k & 1) // PI ?
        ReadText(32); // PI
		  else
        ReadText(33); // Euler
    } 
	  else if(flags1 & F_SHOWPI)
	  {
		   flags1&=~F_SHOWPI;
		   flags&=~F_SHOWINFO;
	  }
}

void TriggerSleepTime()
{
  SleepCnt=user_flags & UF_LONGSLEEP ? 255 : SLEEPTIME;     // keep awake as long as configured
}

void EndInfoText()
{
    flags&=~F_SHOWINFO;
    flags3&=~F_INFOTEXT; // OFF or SLEEP Infotext clear
    flags4&=~F_OFF; // unterbreche GotoSleep falls aktiv
    ShowDisplayNr();
#ifdef MNEMONICS
    if(!(act_flags & AF_RUNMODE)) // this removes also the text SLEEP from display
      ShowMnemonic();
#endif
//	if(user_flags & UF_SYMBOLDISPLAY)
//		ShowPI();
}

// enter Sleep mode, select all rows and activate PORTB Change Interrupt.
// if stopwatch is running activate T1 Overflow Interrupt
void GotoSleep()
{
  GIE=0; // stop display running
  LATD=0; // all segments off
#ifdef SPICE
  LATA=0x83; // select row 7 physical row K1
#else
  LATA=0x43; // select row 7 physical row K7
#endif
  ENABLEGPS=0; // GPS ausschalten

#ifdef PCF2127
  char buf[2];
  buf[0]=0x00;  // normal operation, 24h mode, Second Interrupt disabled
  RTCWrite(0x00,buf,1); // write to Control register 1, disable second interrupt

  if(TMR1ON) // Stopwatch running ?
  {
    TMR1IE=1;  // enable Timer1 interrupt for wakeup from sleep at overflow
  } else
  {
    buf[0]=7; // 5=1024 Hz 6=1 Hz 7=High Z, High Z consumes 0 uA saves power
    RTCWrite(0x0f,buf,1); // write to CLKOUT register
  }
#endif
  TMR2IE=0;  // disable Timer2 Interrupt to avoid wake up by Timer 2 overflow
  TMR2IF=0;  // reset Timer 2 pending bit

L1:
  IOCBN=0x3f; // negative edge interrupts for all 5 columns and /INT for alarm
  IOCBF=0x00; // reset IOC Flags
  IOCIE=1;  // enable Interrupt On Change

// all peripheral interupts must be disabled, because they can wakeup from sleep

#asm
	sleep   ; enter deep sleep mode, FOSC stopped
  nop     ; this operation is executed after sleep
#endasm

#ifdef PCF2127
  if(TMR1IF)  // can only be set if TMR1ON=1 SWSTARTED, every 8,5 Minutes
  {
    TMR1IF=0;

    if(!(hp01flags & F_SWDEC))  // counting upwards
    {
   		SWStartTime++;  // increment overflow counter, add 512 seconds
    } else
    {
      if(SWStartTime==0)  // countdown zero ?
      {
        hp01flags &= ~F_SWDEC; // count upwards after reaching zero
        AlarmCnt=ALARMTIME;
        TriggerSleepTime();
        goto L2;
      } else
   			SWStartTime--;
    }
   	goto L1; // goto sleep again
  }
#endif
	
  if(IOCIF)  // keyboard interrupt occurred or alarm or ON/Off switch?
  {
      if(IOCBF & 1)  // Alarm or TimeStamp interrupt during sleep? Wake up!
      {
				// process Alarm and TS interrupt in DoInterrupt();
      }
      else // keyboard wakeup
      {
        if(!(user_flags & UF_WAKEUPALLKEYS) && (PORTB & 2)!=0)  // only DIV button wakes up
          goto L1; // Goto Sleep again if wrong key pressed

        flags|=F_WAKEUPKEY;  // suppress this keystroke
      }
  }
L2:
  TMR1IE=0; // disable Timer1 interrupt
  IOCIE=0;  // disable Interrupt on Change
  IOCBF=0x00; // reset all IOC Flags, cleares also IOCIF
#ifdef PCF2127
  RTCReadDate();  // read Time before display is activated, if time is displayed
  buf[0]=0x01;  // normal operation, 24h mode, Second Interrupt enabled
  RTCWrite(0x00,buf,1); // write to Control register 1, enable second interrupt
// Wegen Wakeup durch AIF darf das Flag nicht gelscht werden, sonst erscheint keine Alarm Meldung
  buf[0]=5; // 5=1024 Hz
  RTCWrite(0x0f,buf,1); // write to CLKOUT register 1024 Hz
#endif

  flags4 &=~F_OFF;

  EndInfoText();
  ShowLogoInfoText();

  SleepCnt=SLEEPTIME;

  TMR2IE=1;
  PEIE=1;
  GIE=1; // start display running
}


// called every timer tic 8 ms
static void IncrementOperatingTime()
{
	if(++OperatingSeconds>=60) // 1 Minute ?  // oder 6*60 6 Minuten vorbei 1/10 Stunde ?
	{
		OperatingSeconds=0;

    IncrementBCD(OperatingHours,1,0x59);
    if(OperatingHours[0]==0)
			 IncrementBCD(OperatingHours+1,2,0x99); // up to 99 hours

    IncrementBCD(OperatingTime,1,0x59);
    if(OperatingTime[0]==0)
			 IncrementBCD(OperatingTime+1,1,0x99); // up to 99 hours

 	}
}

void ShowOperatingHours()
{
		act_reg[WSIZE-1-0]=OperatingHours[2]>>4;  // show 6 digits operating time
		act_reg[WSIZE-1-1]=OperatingHours[2]&0x0f;
		act_reg[WSIZE-1-2]=OperatingHours[1]>>4;
		act_reg[WSIZE-1-3]=(OperatingHours[1]&0x0f) | 0x80;
		act_reg[WSIZE-1-4]=OperatingHours[0]>>4;
		act_reg[WSIZE-1-5]=OperatingHours[0]&0x0f;

		act_reg[WSIZE-1-7]=OperatingTime[1]>>4;
		act_reg[WSIZE-1-8]=OperatingTime[1]&0x0f | 0x80;
		act_reg[WSIZE-1-9]=OperatingTime[0]>>4;
		act_reg[WSIZE-1-10]=OperatingTime[0]&0x0f;
	  act_reg[WSIZE-1-11]=0x61;  // 'a' shows small letter h
}

void ShowWriteCycles()
{
	act_reg[WSIZE-1-0]=0x0B; // F
	act_reg[WSIZE-1-1]=WriteCycles[2]&0x0f; // show 5 digits Flash WriteCycles 
	act_reg[WSIZE-1-2]=WriteCycles[1]>>4;
	act_reg[WSIZE-1-3]=WriteCycles[1]&0x0f;
	act_reg[WSIZE-1-4]=WriteCycles[0]>>4;
	act_reg[WSIZE-1-5]=WriteCycles[0]&0x0f;
}

// Welcome message anzeigen oder eingeben
void ShowUserStringEntry()
{
  for(uint8_t i=0;i<12;i++)            // show 12-digit string in display
	  act_reg[WSIZE-1-i]=rombuf8[i+StringIndex];

	if(EntryTimerCnt==0)
 	   act_reg[WSIZE-1-ScrollIndex]='_'; // digit cursor ist an bis 512
}

//#define DISPFUNCTION
// DISPFUNCTION indirect function calls cause severe compiler bug with program malfunction because local variables where allocated wrong
// therefore don't use would be only a few bytes shorter
#ifdef DISPFUNCTION

void ShowTime()
{
  timetodsp(act_cl);
}

void ShowWeekday()
{
  ReadText(act_weekday);
}

void ShowHexEntry()
{
 	ShowHex(HexNumber);
}

#ifdef NMEAMENU
static void (* const DispFunction[16])() =
{
ShowDate,ShowTime,ShowAlarmTime,ShowWeekday,ShowTicker,ShowVoltage,ShowTemperature,ShowStopwatch,
ShowHexEntry,ShowOperatingHours,ShowMemoryDetailed,ShowUserFlags,ShowUserStringEntry,ShowLogo,ShowProgramName,NMEAShowData
};
#else
static void (* const DispFunction[16])() =
{
ShowDate,ShowTime,ShowAlarmTime,ShowWeekday,ShowTicker,ShowVoltage,ShowTemperature,ShowStopwatch,
ShowHexEntry,ShowOperatingHours,ShowMemoryDetailed,ShowUserFlags,ShowUserStringEntry,ShowLogo,ShowProgramName,ShowLogo
};
#endif
#endif


void ShowDisplayNr()
{
  if(flags3 & F_INFOTEXT)  // Keine Anzeige der Stopuhr oder Uhrzeit whrend eine Meldung angezeigt wird
    return;

#ifdef DISPFUNCTION
  if(DisplayNr)
  {
    ClearInfo();
    DispFunction[DisplayNr-1] ();
  }
#else
  if(DisplayNr)
  {
    ClearInfo();
    switch(DisplayNr)
    {
    case DISPDATE:    
      ShowDate();
      break;
    case DISPTIME:
      timetodsp(act_cl); //   ShowTime();
      break;
    case DISPALARM:
      ShowAlarmTime();
      break;
    case DISPDAY:
      ReadText(act_weekday); //  ShowWeekday();
      break;
    case DISPTEXT:
     	ShowTicker();
      break;
    case DISPVOLTAGE:
     	ShowVoltage();
      break;
    case DISPTEMP:
     	ShowTemperature();
      break;
    case DISPSTOPWATCH:
     	ShowStopwatch();
      break;
    case DISPHEXENTRY:
 	ShowHex(HexNumber);
//    	ShowHexEntry();
      break;
    case DISPOPERATINGHOURS:
     	ShowOperatingHours();
      break;
    case DISPMEMORY:
     	ShowMemoryDetailed();
      break;
    case DISPUSERFLAGS:
     	ShowUserFlags();
      break;
    case DISPUSERSTRING:
     	ShowUserStringEntry();
      break;
    case DISPLOGO:
     	ShowLogo();
      break;
    case DISPDIRECTORY:
     	ShowProgramName();
      break;
#ifdef NMEAMENU
    case DISPNMEA:
     	NMEAShowData();
      break;
#endif
    }
  }
#endif
}

// invokes User Functions
// nr function number
static void DoFunction(uint8_t nr)
{
	if((nr>=2 && nr<=4) && !(flags & F_PROGRAMMABLE)) // HP-21
    return;
	if(nr==5 && ProgramPages==0 && RegisterPages==0) // dont show Memory map
    return;

	ClearInfo();

	if(nr==DOTKEY)
  {
		ShowRevision();
    flags3|=F_INFOTEXT;
  }
	else if(nr==0)
  {
    DisplayNr=DISPUSERFLAGS;
//    ShowDisplayNr();
		ShowUserFlags();
  }
	else if(nr==1)
  {
    DisplayNr=DISPLOGO;
//    ShowDisplayNr();
		ShowLogo();
  }
#ifdef SHOWROM
	else if(nr==2)
  {
    DisplayNr=DISPROM;
//    ShowDisplayNr();
		ShowROM();
  }
#endif
#ifdef PROGRAMNAME
	else if(nr==2) // Show Programnames Directory
  {
    DisplayNr=DISPDIRECTORY;
    StringIndex=ScrollIndex=0;
    flags3 &=~F_SHOWPROGRAMNR;
//    ShowDisplayNr();
    ShowProgramName();
  }
	else if(nr==3) // Eingabe Programname
  {
    DisplayNr=DISPUSERSTRING;
    ShowProgramName();
    flags1|=F_ENTERPRNAME;
    StringIndex=ScrollIndex=0;
//    ShowDisplayNr();
		ShowUserStringEntry();
  }
#endif
#ifdef PRGMCHECKSUM
	else if(nr==4)
  {
    ShowPRGMChecksum();
    flags3|=F_INFOTEXT;
  }
#endif
#ifdef SHOWMEMORY
	else if(nr==5) // show Memory
  {
    DisplayNr=DISPMEMORY;
   	MemoryPage=0;
    ShowDisplayNr();
//		ShowMemoryDetailed(); // causes compiler error!!! ShowMemotyDetailed@regs has same addres than SST25Read@addr
//    DispFunction[10](); // this uses much more program space
  }
#endif
	else if(nr==6)
  {
    DisplayNr=DISPOPERATINGHOURS;
//    ShowDisplayNr();
		ShowOperatingHours();
  }
	else if(nr==7)
  {
		ShowWriteCycles();
    flags3|=F_INFOTEXT;
  }
	else if(nr==8)
  {
		ShowLEDTest();
    flags3|=F_INFOTEXT;
  }
	else if(nr==9)
  {
    DisplayNr=DISPUSERSTRING;
    flags1&=~F_ENTERPRNAME;
    Readrombuf(TICKERTEXTPAGE,TICKERTEXTADDR); // copy Ticker text to rombuf for editing
//    ShowDisplayNr();
		ShowUserStringEntry();
  }
}


void TransferUserRegisters(uint8_t index,uint8_t write)
{
	uint16_t addr=STOADDR+ HPTypeNr*0x500; // 0x500 per calculator
  addr+=RegisterPages*FPAGESIZE*index;
  for(uint8_t i=0;i<RegisterPages;i++)
  {
    getrampt(i*8);
    if(write)
	    StoreRegisters(REGISTERPAGE,addr,dest,8); // store registers 0-7 8-15 etc
    else
	    RecallRegisters(REGISTERPAGE,addr,dest,8); // recall registers 0-7 or 8-15
    addr+=FPAGESIZE;
  }
}

// Programm nachladen zur Runtime nachdem f ENG instruction erkannt wurde.
// Zur Zeit nur fr HP-25 und HP-29 implementiert.
inline void LoadProgram()
{
  if(ProgramNr<MAXPROGRAMS+USERPROGRAMS)
  {
    ReadProgram(ProgramNr);

    if(HPType==0x29)
    {
	    act_n[1] &=0xF0; act_n[1]|=0x07; act_n[0] =0x2D; // set program step 01 coded as 62D, 7 will be decremented
	    act_pc=07055;  // skip ENG instruction, proceed at 7055
    }
    else
    {
	    act_m[1] &=0x0F; act_m[1]|=0x10; act_m[2] &=0xF0; // set program step 01
	    act_pc=01743;  // skip ENG instruction, proceed at 1743
    }
  }
}

// wird aufgerufen nach Ausfhrung von op_gokeys, op_keys_to_a, op_a_to_romaddress, op_return
// um Sonderaktionen whrend der Emulation auszufhren.
void CheckAddress()
{
  uint16_t addr=act_pc;

// Um zu warten bis ein Tastenbefehl vollstndig ausgefhrt ist, mus das Flag WAITKEY2ROM gesetzt werden.
// Danach wird in zwei Stufen getestet, dass keys to rom ausgefhrt und die Idle Schleife erreicht wurde.

	if(opcode==0020 || opcode==0120)  // wait for keys_to_romaddress or keys_to_a until idle loop is left after key pressed
	{
	  if(flags2 & F_WAITKEY2ROM)
	  {
			flags2&=~F_WAITKEY2ROM;
			flags2|=F_WAITIDLE; // switch off display until calculaton is finished
		}
	}

#ifndef SPICE

  if(HPType==0x25)
  {
#ifdef HP82240
  	if(user_flags & UF_HP82240)
	  {
		   if(addr==02025)  // g NOP is Paper Advance command
			    SendHP82240LineFeed();
    } 
#endif
// addr 01427 wird angesprungen beim Eingeben von f ENG 0-9 oder beim ausfhren mit SST und R/S und manuell
// in act_a[0..1] steht der Programmcode 70-79, act_a[0] ist die Anzahl der Dezimalstellen
// SST or Program running f ENG, code  77,78,79 in a[0..1]

    if(addr==01427 && act_flags & AF_RUNMODE && act_a[1]==7 && act_a[0]>=7 && act_a[0]<=9  ) // LoadProgram only in RUN mode either if program is running or manual
      goto L1;

    if(addr==01436 && act_a[1]==0x0E && act_a[0]==0x0C) // g E+ is programmed sound beep, program step code 0xEC in act_a[1..0]
    {
        SoundBeep();
        BeepCnt=100; // 800 ms Beep
    }
  }

  if(HPType==0x29 && act_flags & AF_RUNMODE )
  {
    if(addr==0430 && (act_s & 1<<4) && !(act_s & 1<<6)) // addr==1314 g E+ in RUN mode is sound beep, 1312 also in PRGM mode
    {
      SoundBeep();
      BeepCnt=100; // 800 ms Beep
    }
// this address will be arrived after a return instruction
    if(addr==07044 && act_flags & AF_RUNMODE && act_a[1]==6 && act_a[0]>=7 && act_a[0]<=9) // f ENG 7-9 Programm nachladen
    {
L1:;
#ifdef LOADPROGRAM
  	  if(act_a[0]==9) // ENG 9 load program indexed by Register 2
    	{
	      getrampt(2); // dest= act_ram[2] register 2
        src=dest;
        dest=act_reg;
        reg_decompress();
      	ProgramNr=ConvertBCD2Hex(act_reg);
      }
      else if(act_a[0]==8) // ENG 8 load previous program
      {
   		  if(ProgramNr==0)
          ProgramNr=MAXPROGRAMS+USERPROGRAMS-1;
        else
          ProgramNr--;
      }
   	  else // ENG 7 load next program
      {
   		  if(++ProgramNr>=MAXPROGRAMS+USERPROGRAMS)
     		  ProgramNr=0;
      }
      LoadProgram();
#endif
    }
  }
#endif
}

#ifdef STOPWATCH
static void EnterStopwatch()
{
	DisplayNr=DISPSTOPWATCH;
  ClearInfo();
 	ReadStopwatch();
}
#endif

#ifdef NMEAMENU
static void EnterNMEA()
{
	DisplayNr=DISPNMEA;
  NMEAEnable();  // enable 3.3V power for GPS module
  NMEADoKeys(-1); // reset Timeoutcnt and show data
}
#endif

void StopShowRegister()
{
  if(keyflags & KF_SHOWREGISTER || keyflags & KF_SHOWPROGRAM)
  {
    keyflags&=~KF_SHOWPROGRAM; // stop showing program or register
    keyflags&=~KF_SHOWREGISTER;
	  keyflags|=KF_ALLOWKEY; // allow keys after supressing this key
	  flags&=~F_SHOWINFO; // this is only necessary for SHOWREGISTER
  }
}

void CheckProgramEmpty()
{
	if(GetPRGMChecksum()!=0)
		crc_flags&=~(1<<4); // if program is empty activate math function keys
	else
		crc_flags|=(1<<4);
}

static void SwitchPrgmRun()
{
//	EnterCalculator();

  if(HPType!=0x38) // 38 uses key for switching between RUN/PRGM
  {
	  act_flags&=~AF_RUNMODE;
    if((flags & F_SLIDERRIGHT) || !(flags & F_PROGRAMMABLE))
		  act_flags|=AF_RUNMODE;
  }

#ifdef MNEMONICS
  flags&=~F_SHOWMNEMONIC;
  if(!(act_flags & AF_RUNMODE) && !(TypeFlags & TF_CLASSIC))
    ShowMnemonic();
#endif
  
	flags2&=~F_WAITKEY2ROM;
	flags2&=~F_WAITIDLE; // end wait for idle
  StopShowRegister();
	TriggerSleepTime(); // dont sleep
  if(HPType==0x67)
  	CheckProgramEmpty();
}

#ifdef EXTENDEDFEATURES

// Fhre extended Funktion aus
// action: Nummer der auszuführenden Aktion
inline void DoKeyAction(uint8_t action)
{
uint8_t a;

			switch(action)
			{
#ifdef NMEAMENU
				case 1:
          if(DisplayNr==DISPNMEA)
          {
           	keyflags|=KF_CLEARPREFIX|KF_ALLOWKEY;
            EnterCalculator();
          }
          else
					  EnterNMEA();
					break;
#endif
				case 2:
					flags|=F_SLIDERRIGHT;
					SwitchPrgmRun();
					break;
				case 3:
					flags&=~F_SLIDERRIGHT;
					SwitchPrgmRun();
					break;
#ifdef HEXENTRY
				case 4:
          if(!(act_s & RunBit)) // only if program not running
          {
          	if(DisplayNr==DISPHEXENTRY)
          	{
          		LeaveHexEntry();
            	EnterCalculator();
          	}
          	else
            {
					    DisplayNr=DISPHEXENTRY;
 	  					keyflags|=KF_CLEARPREFIX;
  	          keyflags|=KF_ALLOWKEY;
					  	EnterHexEntry();
            }
					}
					break;
#endif
#ifdef STOPWATCH
				case 5:
          if(DisplayNr==DISPSTOPWATCH)
          {
            hpcode=0; // this avoid to stop a running program with hpcode 0x74
            EnterCalculator();
//	          keyflags|=KF_SUPRESSKEY;
//	          keyflags|=KF_ALLOWKEY;
          }   
          else
					  EnterStopwatch();
					break;
#endif
				case 6: // free
					break;
#ifdef UNDO
				case 7:
					UndoStack();
					break;
				case 8:
					UndoProgramStep();
					break;
#endif
				case 9:
					keyflags|=KF_SHOWPROGRAM; // must be set separately because not within uint8_t range 1<<8
					RepeatTime=REPEATTIME*2; // GetConstant(21)*2;
          if(act_flags & AF_RUNMODE)
             RepeatTime*=2; // double Time in RUN Mode
					flags3|=F_FIRSTSTEP;
					break;
				case 10:
					DeleteProgramStep();
					break;
				case 11:
					InsertProgramStep();
					break;
				case 12:
					act_press_key(GetIndexKeyCode(ENTERKEY)); // press ENTER to lift stack or because f or g was pressed before 
					// no break;
				case 13:
					TimerCnt=-20; // wait for ENTER release
					break;

				case 14:
          if(DisplayNr==DISPCALCULATOR)
          {
            if(!(TypeFlags & TF_SPICE)) // spice calculators does have built in ShowMantissa except HP37/38
	            ShowMantissa();
#ifdef HP82240
						if(user_flags & UF_HP82240)
							SendHP82240Display();
#endif
          }
					break;
				case 15:
#ifdef HP82240
					if(user_flags & UF_HP82240 && act_flags & AF_RUNMODE)
						SendHP82240LineFeed();
#endif
#ifdef SHOWMEMORY
          if(RegisterPages || ProgramPages)
          {
            ShowMemory();
            flags3|=F_INFOTEXT; // wird ausgeblendet nach einigen sekunden
          }
#endif
          break;
        case 16:
          ReadText(9); // SLEEP
          flags3|=F_INFOTEXT; // Text wird quittiert mit nchstem Tastendruck
          flags4|=F_OFF; // aktion kann noch mit Taste abgebrochen werden
          SleepCnt=2;
          break;
#ifdef PRGMCHECKSUM
        case 17:
          ShowPRGMChecksum();
          break;
#endif
        case 18:
        case 19:
        case 20:
        case 21:
#ifdef PCF2127
            RTCReadDate(); // vor Anzeige der Uhrzeit aktuelle Uhrzeit lesen
#endif
        case 22:
        case 23:
        case 24:
          Quotation=0;
L1:
          TickerIndex=0;
          DisplayNr=action-18+1;
          ShowDisplayNr();
          break;
        case 25:
          Quotation=TMR2Cnt & 0x7FFF; // 32k
          do
          {
            a=GetTickerText(0); // find beginning of next quotation
            Quotation++;
          } while(a!=TICKEREOT);
          action=22;
          goto L1;
      }
}

inline uint8_t CheckKeySequence()
{
  uint8_t pt[8];
	uint8_t i,mode,action;
  for(i=0;;i++)
	{
    SST25Read(KEYCODEPAGE, KEYSEQUENCE+i*8,pt, 8); // read from Keyseqeunce Table

		if(pt[0]==0xFF)  // End of Table
      break;

    if(keyindex!= (pt[5] & 0x1f)) // same keyindex found ?
      continue;
    
		if(pt[0]!=HPType && pt[1]!=HPType && pt[2]!=HPType && pt[3]!=HPType) // appropriate calculator
      continue;

		if(!(act_flags & AF_RUNMODE && pt[4] & 1 || !(act_flags & AF_RUNMODE) && pt[4] & 2)) // correct mode ?
      continue;

    mode=pt[5] & 0xe0; // check prefix

		if(!(	mode==0x20 && prefixflags & PF_FKEY || mode==0x40 && prefixflags & PF_GKEY || mode==0x60 && prefixflags & PF_HKEY || mode==0 && !(prefixflags & (PF_FKEY|PF_GKEY)) || // correct prefix key or no keys?
 			mode==0xa0 && prefixflags  & PF_FFKEY || mode==0xc0 && prefixflags   & PF_GGKEY))  // f f  or g g
      continue;

	  keyflags|=pt[6];
		action=pt[7];
		if(action & 0x40)
    {
		  TimerCnt=0;
      RepeatCnt=0;
    }
		if(action & 0x80)
		  key=keycode;
		DoKeyAction(action & 0x3f);
		classicflags&=~F_FKEYPRESSED; // dont call FKEYRELEASED if f key was part of KeyAction
		return TRUE;
  }
  return FALSE;  
}

#endif // EXTENDEDFEATURES

#ifdef PROGRAMNAME
void ShowProgramName()
{
  if(flags3& F_SHOWPROGRAMNR)
  {

//    ShowCheckSum(ProgramNr); // causes compiler problem wrong variable addresses because ShowProgramName is called indirect
    ReadText(31); // Program
    DisplayDecimal(ProgramNr,4);
  }
  else
  {
  uint16_t addr=ProgramAddr+ProgramPages*ProgramNr*FPAGESIZE; // add Base Address of Prgoram area of this calculator
  addr+=ProgramPages*FPAGESIZE-12;
	SST25Read(LibraryPage,addr,rombuf8,12); //  read Program Name to rombuf stored in 12 bytes at end of Program Pages
  ClearInfo();
#ifdef SPICE
  for(uint8_t i=0;i<11;i++) // Spice calculators have 11 digits
    act_reg[WSIZE-2-i]=rombuf8[i];
#else
  for(uint8_t i=0;i<12;i++) // Woodstock calculators have 12 digits
    act_reg[WSIZE-1-i]=rombuf8[i];
#endif
  }
  if(!ProgramEmpty(ProgramNr))
#ifdef SPICE
    act_reg[WSIZE-1-(MAXDIGITS-2)]|=0x80;
#else
    act_reg[WSIZE-1-(MAXDIGITS-1)]|=0x80;
#endif
}

#endif

// Store Program 00-99 f STO . nn, and 0-9 f STO n
// n=0 bis 99,   n=MAXPROGRAMS bis MaxPrograms+9
void StoreProgram(uint8_t n)
{
  ProgramNr=n;
	WriteProgram(n);
#ifdef PRGMCHECKSUM
	ShowPRGMChecksum();
#endif
}

void RecallProgram(uint8_t n)
{
    ProgramNr=n;
    ReadProgram(n);
#ifdef PROGRAMNAME
    flags3 &=~F_SHOWPROGRAMNR;
    ShowProgramName();
    flags3|=F_INFOTEXT;
#endif
#ifdef PRGMCHECKSUM
    if(HPType==0x67)
      CheckProgramEmpty();
#endif
}

// execute Dot Command STO . 00-99. RCL . 00-99, GTO . 00-49
void DotCommand()
{
  if(keyflags & KF_GTOSTEP) // GTO program step ?
	{
    if(!(TypeFlags & TF_SPICE) && !(TypeFlags & TF_CLASSIC))
	   SetProgramStep(keydigit);
	}
	else // Recall or Store Library program or constant
	{
  	if(act_flags & AF_RUNMODE)
    {
        uint8_t *pt=act_n; // HP-25
        if(HPType==0x33)
           pt=act_m;
        if(TypeFlags & TF_21_29_27) // HPType==0x21 || HPType==0x29 || HPType==0x27)
           pt=act_reg;

			  if(keyflags & KF_RCLPRGM) // f RCL . load constant ?
			  {
          RecallConstant(pt, keydigit);
			  	if(pt==act_reg)
				  {
					  src=act_reg; dest=act_c;
					  reg_decompress();
					  keyflags&=~KF_CLEARPREFIX;
					  act_press_key(GetIndexKeyCode(GKEY)); // press g ENTER to normalize Entry and stack lift
				    flags2|=F_PRESSENTER;
				    flags2|=F_WAITKEY2ROM;
				  }
		    }
		    else
		    {
				  if(pt==act_reg)
				  {
					  src=act_c;
            dest=act_reg;
					  reg_compress();
				  }
			    StoreConstant(pt, keydigit);
			  }
    }
    else // PRGM
    {
      if(HPType==0x67 || HPType==0x34)
      {
				  if(keyflags & KF_RCLPRGM)
          {
            if(flags2 & F_DOUBLEPREFIX)
            {
              if(keydigit>=16) // 16 Standard Pac programs or 16 HP-34 Programs
                goto L1;

              keydigit+=110;
            }
					  RecallProgram(keydigit);
            if(HPType==0x34)
            {
            uint8_t *pt=(uint8_t *)ACT_RAM+47*7; // same function in RUN and PRGM mode, because programs and registers are dynamically mixed
						pt[0]=0;  // set programstep to 000
						pt[1]=0;  // set programstep to 000
            }
          }
				  else 
					  StoreProgram(keydigit);
      }
      else
      {
			  if(keyflags & KF_RCLPRGM)
        {
            if(flags2 & F_DOUBLEPREFIX && HPType==0x25)
            {
              if(keydigit<HP25APPLICATIONPROGRAMS)
					      RecallProgram(keydigit+110);  // load HP-25 Applicationprogram without checksum
              else
              {
L1:
				        ShowText(55);
              }
            }
            else
            {
					    RecallProgram(keydigit);  // load 7 registers, up to 100 programs
            }
        }
				else
				 StoreProgram(keydigit);
		  }
	  }
  } 
}

// Wird angesprungen bei keypress wenn Zeit Datum oder Alarm oder Stopuhr angezeigt wird
uint8_t DoTimeEntry()
{
uint8_t ReturnCode=1;
uint8_t *pt,i;

  if(DisplayNr==DISPDIRECTORY)
  {
    if(keyindex==STOKEY)
    {
      if(ProgramNr<MAXPROGRAMS+USERPROGRAMS) // don't overwrite fixed programs
        StoreProgram(ProgramNr);
      goto L2;
    }
    if(keyindex==RCLKEY)
    {
      ReadProgram(ProgramNr);
      goto L2;
    }
  }
  if(keyindex==ENTERKEY)
  {
    if(DisplayNr==DISPUSERSTRING)
    {
      if(flags1 & F_ENTERPRNAME)
      {
        uint16_t addr=GetProgramAddr(ProgramNr+1)-64; // add Base Address of Program area of this calculator
        src=rombuf8; dest=rombuf8+64-12;
        first=0; last=12-1;  reg_copy();
        SST25Read(LibraryPage,addr,rombuf8,64-12); //  read Program Name to rombuf stored in 12 bytes at end of Program Pages
        Writerombuf(LibraryPage,addr);
      } else
      {
        rombuf8[63]=TICKEREOT; // prohibit endless loop
        Writerombuf(TICKERTEXTPAGE,TICKERTEXTADDR);
      }
    }
    
    if(DisplayNr>DISPDAY && DisplayNr!=DISPSTOPWATCH && DisplayNr!=DISPNMEA)
    {
      if(DisplayNr==DISPLOGO && DispHPTypeNr!=HPTypeNr)
      {
        keyindex=DispHPTypeNr;
        SwitchCalculator();
      }
L2:
      EnterCalculator();                     // ENTER schaltet zurck auf Calculator Mode
      keyflags|=KF_SUPRESSKEY|KF_ALLOWKEY;
      return 1;
    }
  }

  if(DisplayNr==DISPTIME) // Time displayed?
    pt=act_cl;
  else if(DisplayNr==DISPDATE) // Date displayed?
    pt=&act_cl[6];
  else if(DisplayNr==DISPALARM) // Alarm displayed?
    pt=act_al;
  else
    pt=act_sw;

  if(keyindex==CLXKEY)
  {
      if(entrydigit)
      {
        entrydigit=0;
        keyindex=0;
        goto L3;
      } else
      {
         if(DisplayNr==DISPLOGO)
           DispHPType=HPType; // auf alten Wert zurcksetzen

         if(DisplayNr!=DISPSTOPWATCH && DisplayNr!=DISPHEXENTRY && DisplayNr!=DISPNMEA) //        CLX nicht bei Stopwatch sonst wird Stopuhr nicht zurckgesetzt
         {
           EnterCalculator();                     // CLX schaltet zurck auf Calculator Mode
           keyflags|=KF_SUPRESSKEY|KF_ALLOWKEY;
         } else
           return 0;
      }
  }
  if(DisplayNr==DISPTEXT)
  {
         switch (keyindex)
         {
          case PLUSKEY:   // schneller
            TickerSpeed=TickerSpeed*3/4;
            if(TickerSpeed<10)
              TickerSpeed=10;
            break;
          case MINUSKEY: // langsamer
            TickerSpeed=TickerSpeed*4/3;
            if(TickerSpeed>150)
            {
              TickerSpeed=150;  // 150 ist Stillstand
              TickerIndex=0;
              ShowTicker();
            }
            break;
        }
        return TRUE;
  }

  if(DisplayNr<DISPTEXT || DisplayNr==DISPSTOPWATCH) // Time date alarm weekday stopwatch
  {
    if(keyindex==CHSKEY)
    {
       if(DisplayNr==DISPTIME || DisplayNr==DISPALARM) // Uhrzeit angezeigt
         user_flags^=UF_TIMEAMPM;
       if(DisplayNr==DISPDATE) // Datum angezeigt
         user_flags^=UF_DATEUS;
    }
    
    if(keyindex<10)
    {
      if(DisplayNr==DISPDAY)  // wekkday cannot enter entrymode, entrydigit never set
      {
        act_weekday=keyindex;
        if(act_weekday>6) act_weekday=6; 
#ifdef PCF2127
        RTCWriteDate();
#endif
        goto L1;
      }

      entrydigit++;
      if(entrydigit==1)
      {
L3:
         flags|=F_TIMEENTRY;
         for(i=0;i<6;i++)
           pt[i]=0;
         if(DisplayNr==DISPSTOPWATCH) // Stopwatch
         {
           pt[6]=0;  // stopwatch has 8 digit including HH:MM:SS:TT  hours minutes seconds ticks
           pt[7]=0;
         }   
      }
      if(entrydigit<=6)
      {
         if(entrydigit & 1)  // erstes Zeichen in Einer eintragen
           pt[5-entrydigit]=keyindex;  // erstes Zeichen ist pt[4] Eingabe von links nach rechts 
         else
         {
           pt[7-entrydigit]=pt[6-entrydigit];  // Einer nach Zehner schieben 
           pt[6-entrydigit]=keyindex;          // zweites Zeichen in Einer eintragen
         }
      }
    }
    else if(keyindex==ENTERKEY)
    {
      flags&=~F_TIMEENTRY;  // end time entry, show actual time
			if(entrydigit || DisplayNr==DISPALARM)  // Eingabe abschlieen, // Alarm toggeln wenn keine Eingabe
      {
        if(DisplayNr==DISPSTOPWATCH)
        {
           TMR1Reset(GetSWTime());
        }
#ifdef PCF2127
        else if(DisplayNr==DISPALARM)
        {
          if(hp01flags & F_ALARMACTIV)  // Einschaltreihenfolge: aktiv, aktiv tglich, inaktiv
          {
            flags4^=F_ALARMDAILY;
            if(!(flags4 & F_ALARMDAILY))
              hp01flags &=~F_ALARMACTIV;
          }
          else
            hp01flags|=F_ALARMACTIV;

          RTCSetAlarm(hp01flags & F_ALARMACTIV);
        }
        else
          RTCWriteDate();
#endif

        goto L1;   
      }
    }
    else  // Men verlassen wenn keine gltige Taste
    {
      ReturnCode=0;
L1:
      flags&=~F_TIMEENTRY;  // end time entry, show actual time
      entrydigit=0;
#ifdef PCF2127
      RTCReadDate(); // Display aktualisieren
#endif
    }
    ShowDisplayNr();
  }
  return ReturnCode;
}

void StartRepeat()
{
	keyflags|=KF_REPEATKEY;
	RepeatTime=REPEATTIME; // GetConstant(21);
  RepeatDelayTime=0;
	TimerCnt=RepeatCnt=0;
}

// Wird aufgerufen nach keypress wenn das Prefix gelscht werden soll
void ClearPrefix()
{
	keyflags&=~KF_CLEARPREFIX;
	prefixflags=0;

//  if(HPType==0x33)
//    act_s|=(1<<6); // change g prefix to f prefix before pressing ENTER, f = s4  g = s4+6

  if(HPType==0x34)
  {
	  act_s&=0xbf7f; // reset s7, s14 prefix  p=9 is h prefix  p=12 is no prefix
    if(act_flags & AF_RUNMODE) // wenn in Idle Loop, springe direkt nach Convert, dann wird act_n neu angezeigt. zB nach RecallConstant
      act_pc=03412; // act_p=9; // simuliere h key pressed
    else
      return;
  }

  if(HPType==0x37)
	  act_s&=~(1<<13); // reset s13 prefix
  else if(HPType==0x67)
	  act_s&=~0x21d0; // reset s13 prefix, s8,s7,s6,s4 prefix qualifiers
#ifdef HP01
  else if(HPType==0x01)
    act_s&=~(1<<4);
#endif
  else
  {
    uint8_t code=GetIndexKeyCode(ENTERKEY);
    act_press_key(code); // press ENTER to clear prefix because f or g was pressed before
    keypressed&=~KP_KEYDOWN; // don't let change the keycode while another key is still pressed
//    keycode=code;
//    keypressed|=KP_KEYDOWN; // will call act_key_press with keycode
  }
}

inline void DoKeys()
{
		if(!(TypeFlags & TF_SPICE) && !(keyflags & KF_SUPRESSKEY)) // do actions to be done for many keys like undo and rightadjust
		{
#ifdef UNDO
			if(act_flags & AF_RUNMODE)
			{
				if(keyindex<10 || keyindex==PLUSKEY || keyindex==MINUSKEY || keyindex==MULKEY || keyindex==DIVKEY)  // save stack
          if(ProgramPages<4)  // RamBanks <=6, this excludes HP-34 HP-67, otherwise not enough RAM space for undo_stack
					  SaveStack(); 
			}
			else if(keyindex!=FKEY && keyindex!=GKEY && keyindex!=bstkey) // save program step
				SaveProgramStep();
#endif

#ifdef RIGHTSHIFT
			if((user_flags & UF_SYMBOLDISPLAY || user_flags & UF_RIGHTADJUST) && act_flags & AF_RUNMODE &&	keyindex!=sstkey && keyindex!=bstkey && keyindex!=RSKEY)
			{
				flags2|=F_WAITKEY2ROM;
				TimerCnt=0;
			} else
				rightshift=0; // no rightshift SST BST RS shows programstep temporarily, use else to avoid flicker
#endif
		}

    StopShowRegister();

    if(DisplayNr==DISPNMEA && prefixflags & PF_FKEY && keyindex==STOKEY)
     goto L1;
 
		if(CheckKeySequence())
		{
       if(keyflags & KF_RCLPRGM)
       {
         if((HPType==0x25 && prefixflags & PF_FFKEY) || (HPType==0x67 && prefixflags & PF_GGKEY)) // only HP-67 recall standard pac
           flags2|=F_DOUBLEPREFIX;  // merke double prefix bis Commando abgeschlossen.
       }
         
       if(act_s & RunBit) // wenn Programm luft dann bestimmte Befehle nicht ausfhren
         StopShowRegister();

#ifdef HP01
       if(HPType==0x01)
       {
          if(keyindex!=0)
          {
  				  keyflags|=KF_ALLOWKEY;
	  			  keyflags|=KF_CLEARPREFIX;
          }
       }
#endif
      goto L2;
		}
L1:
    if(keyflags & KF_FUNCTION) // skip unnecessary if then
      goto L3;

    if(DisplayNr && DisplayNr!=DISPHEXENTRY && prefixflags==0) // if(DisplayNr & 0x1f)>=DISPDATE && (DisplayNr & 0x1f)<=DISPSTOPWATCH)
    {
      DoTimeEntry();
    }

		if(DisplayNr==DISPDIRECTORY)
    {
	    EntryTimerCnt=0;
      DoUserStringEntry();  // execute key and repeat after time
    }
		
		if(DisplayNr==DISPUSERSTRING)
		{
      uint8_t c,i;
    	char *pt=(char *)rombuf;

	    EntryTimerCnt=0;

      c=pt[ScrollIndex+StringIndex];

			if(keyindex==DOTKEY) // dezimalpunkt toggle
				pt[ScrollIndex+StringIndex]=c^0x80;

      if(hpcode==0x13 /*GTOKEY*/)
	  	  pt[ScrollIndex+StringIndex]=TICKEREOT; // set end of text marker with GTO key

			for(i=0;i<ALPHAKEYS;i++) // teste alpha tasten
			{
				if(hpcode==AlphaKey[i])
			  {
          i=AlphaDigit[i];
					goto L5;
				}
			}
	
			if(keyindex<10) // Ziffer von 0-9 ?
			{
          i=keyindex|'0'; // make ASCII character from number 0-9
L5:			  pt[ScrollIndex+StringIndex]=(c & 0x80) | i; // Dezimalpunkt beibehalten
          hpcode=0x11;  // Cursor nach rechts
          DoUserStringEntry();
			}
		}

    if(DisplayNr==DISPUSERFLAGS)
    {
#ifdef SPICE
      if(keyindex==9) // Flag 9 kann nicht angezeigt wernde, eshalb berspringen
        goto LUF3;
#endif
      if(keyindex<10)
      {
        if(keyindex==5) // wenn bit 5 Rightadjust abgeschaltet wurde, dann Anzeige wieder linksbndig
          rightshift=0;
        goto LUF1;
      }

			if(keyindex<=DIVKEY) // erlaube toggle Flag 19 auch wenn SPICE Flag 19 nicht anzeigen kann
      {
		    user_flags1^=1<<(keyindex-10);
        keyindex+=6;
        goto LUF2;
      }

		  for(uint8_t i=0;i<6;i++)  // test A-F before keyindex<10
		  {
			  if(hpcode==AlphaKey[i])
			  {
				  keyindex=i+=10; // toggle user_flags A-F
	   			goto LUF1;
			  }
		  }
      goto LUF3;
LUF1:
		  user_flags^=1<<keyindex;
LUF2:
      KeyDownCnt=0;
      TraceCode=keyindex+34; // Text Index to show
LUF3:
      ShowDisplayNr(); // Anzeige aktualisieren
		}

		if(DisplayNr==DISPLOGO)
		{
      if(keyindex==PLUSKEY && DispHPTypeNr<NROFCALCULATORS-1)
      {
        DispHPTypeNr++;
        goto SL1;
      }
      if(keyindex==MINUSKEY && DispHPTypeNr>FIRSTCALCULATOR)
      {
        DispHPTypeNr--;
      }
SL1:
      ReadFlashInfo(0);
      if(rombuf8[32+DispHPTypeNr]==0) // ROM code available ?
        ReadText(28); // NO ROMCODE
      else
      {
        DispHPType=rombuf8[DispHPTypeNr];
        ShowLogo();        
      }
    }  
		if(DisplayNr==DISPOPERATINGHOURS)
		{
		  if(keyindex==0)
			{
				OperatingTime[0]=0;
				OperatingTime[1]=0;
        ShowDisplayNr();
			}
		}

#ifdef SHOWROM
		if(DisplayNr==DISPROM)
		{
			EntryTimerCnt=0;

			if(keyindex==CHSKEY)
      {
				flags1^=F_SHOWHEX;
      }
			else if(keyindex==PLUSKEY || keyindex==MINUSKEY || hpcode==0x11 || hpcode==0x12) // SST BST
      {
        DoShowROM();
				EntryTimerCnt=100; // start RepeatDelayTime
      }
      ShowDisplayNr();
		}
#endif
#ifdef SHOWMEMORY
		if(DisplayNr==DISPMEMORY)
		{
			if(keyindex==CHSKEY)
      {
        flags4^=F_SHOWPRGM;
        MemoryPage=0;
      }
			else if(keyindex==PLUSKEY)
      {
        if(MemoryPage<10)
          MemoryPage++;
      }
			else if(keyindex==MINUSKEY && MemoryPage>0)
        MemoryPage--;
      ShowDisplayNr(); // Anzeige aktualisieren
    }  
#endif
#ifdef STOPWATCH

    if(DisplayNr==DISPSTOPWATCH) // Stopwatch enabled ?
			DoStopwatchKeys();
#endif
#ifdef HEXENTRY
		if(DisplayNr==DISPHEXENTRY)
			DoHexEntry();
#endif
#ifdef NMEAMENU
		if(DisplayNr==DISPNMEA) // NMEA enabled ?
			NMEADoKeys(keyindex);
#endif
		if(DisplayNr==0)
		{
#ifdef HP82240
			if(user_flags & UF_HP82240 && user_flags & (UF_PRINTNORM|UF_PRINTTRACE) && act_flags & AF_RUNMODE && !(keyflags & KF_SUPRESSKEY)) // trace only when activated and in RUN mode
				CheckTraceKey();
#endif
L3:
			if(keyflags & KF_FUNCTION && (keyindex<10 || keyindex==DOTKEY)) // g GTO invokes Functions 0-9 .
			{
				DoFunction(keyindex);
				keyflags|=KF_ALLOWKEY;
			}
			else if(keyindex==DOTKEY && (keyflags & (KF_RCLPRGM|KF_STOPRGM)))
			{
				keyflags|=KF_DOT|KF_FIRSTDIGIT|KF_SUPRESSKEY|KF_CLEARPREFIX;
			}
		  else if(HPType==0x25 && keyindex==DOTKEY && prefixflags & PF_GTOKEY /*&& !(act_flags & AF_RUNMODE)*/)   // GTO . nn jumps to program step only HP-25
			{
				keyflags|=KF_GTOSTEP|KF_SUPRESSKEY|KF_DOT|KF_FIRSTDIGIT;
			}
			else if((keyflags & (KF_STOPRGM|KF_RCLPRGM|KF_GTOSTEP|KF_FUNCTION|KF_DOT)) && keyindex>=10)  // falsche Taste nach f STO oder f RCL
			{
				keyflags|=KF_ALLOWKEY;
				keyflags&=~(KF_STOPRGM|KF_RCLPRGM|KF_GTOSTEP|KF_FUNCTION|KF_DOT);
			}
			else if(keyflags & KF_DOT) // GTO .  RCL . STO .
			{
				if(!(keyflags & KF_FIRSTDIGIT)) // second digit ?
				{
					keydigit*=10;
					keydigit+=keyindex;
					DotCommand();

					keyflags &=~KF_DOT;  // end of DOT command
					keyflags|=KF_ALLOWKEY; // keystrokes allowed again now
				}
				else
				{
					keyflags&=~KF_FIRSTDIGIT;
					keydigit=keyindex; // store first digit
				}
			}
			else if(keyflags & KF_STOPRGM)
			{
		    if(keyindex<USERREGISTERS)
        {
          if(act_flags & AF_RUNMODE)
          {
            TransferUserRegisters(keyindex,TRUE);
            if(HPType==0x27)
      				keyflags&=~KF_CLEARPREFIX; // bug, emulator stall after act_press_Key(ENTERKEY) code 0xd3
          }
          else
					  StoreProgram(keyindex+MAXPROGRAMS);
        }     

				keyflags|=KF_ALLOWKEY;  // will also clear F_STOPRGM
			}
			else if(keyflags & KF_RCLPRGM)
			{
				if(keyindex<USERREGISTERS && act_flags & AF_RUNMODE) // f RCL read RAM Bank
				{
          TransferUserRegisters(keyindex,FALSE);
				}
				if(keyindex<USERPROGRAMS && !(act_flags & AF_RUNMODE)) // f RCL read RAM Bank
				{
					RecallProgram(keyindex+MAXPROGRAMS); // recall program
				}
				keyflags &=~KF_RCLPRGM;
				keyflags|=KF_ALLOWKEY;
			}
			else if(user_flags & UF_REPEAT)
      {
         if(!(act_flags & AF_RUNMODE) && (keyindex==sstkey || keyindex==bstkey))
			   {
            if((HPType==0x34 && prefixflags & PF_HKEY) || prefixflags==0)
              StartRepeat();
        }
        if((act_flags & AF_RUNMODE) && (keyindex==ROLLKEY || keyindex==XYKEY || keyindex==PLUSKEY || keyindex==MULKEY))
			  {
            if(prefixflags==0) // repeat nur wenn kein prefix
              StartRepeat();
			  }
		  }
    }  
L2:
    CheckPrefixKeys();
}


// Check if key pressed execute keycode, must be called every instruction cycle
//inline void CheckKeyPress()
void CheckKeyPress()
{
#ifdef SERIALCOMMAND
  if(RemoteKey)
  {
#ifdef ICDEBUG // In Circuit Debugging
    if(RemoteKey >= 0x80) // Debug SingleStep Command
    {
      DebugCommand=RemoteKey & 0x7f;
      RemoteKey=0;
      return;
    }  
#endif

    hpcode=RemoteKey;
    RemoteKey=0;
    flags|=F_REMOTEKEY;
    goto LR1;
  }
#endif
  if(keypressed & KP_PRESSED) // key pressed ?
	{
		keypressed&=~KP_PRESSED;
#ifdef SERIALCOMMAND
LR1:
#endif
    keypressed|=KP_KEYDOWN;
    LongPressCnt=0;
    KeyDownCnt=0;

    flags2&=~F_PREFIX;
    if(prefixflags)
      flags2|=F_PREFIX; // this bit will be remain one button after prefix 

		keycode=GetKeyCode(hpcode);
		keyindex=GetKeyIndex();  // hpcode setzen fr Classic und andere

//    flagsx|=F_READKEYS; // signal keycode is available for any use

    TriggerSleepTime();

#ifdef SHOWKEY
    if(user_flags1 & UF_SHOWKEY && act_flags & AF_RUNMODE && DisplayNr==DISPCALCULATOR /*&& !(prefixflags & (PF_FFKEY| PF_GGKEY))*/)
      TraceCode=GetTraceCode(TRACETABLE);
#endif

    if(AlarmCnt)  // beende Alarm
    {
      AlarmCnt=0;
      EndSoundBeep();
    }

	  if(flags3 & F_INFOTEXT) // Wird eine Meldung angezeigt ? Error, Sleep, Alarm etc.
    {
#ifdef HPSERIALNUMBER
      if(flags3 & F_SHOWSERIAL) // show two info texts after each other ?
      {
        flags3&=~F_SHOWSERIAL;
        ShowSerialNumber();
      }
      else
#endif
      EndInfoText();
      keypressed&=~KP_KEYDOWN;
      return;
    }

    if(keyindex==FKEY && !(flags & F_REMOTEKEY) && (/*TypeFlags & TF_CLASSIC ||*/ HPType==0x67)) // f key pressed or released
    {
        if(!(classicflags & F_FKEYPRESSED)) // ignore f key pressed and process if released
        {
		      classicflags|=F_FKEYPRESSED|F_FKEYDOWN;  // mark f key pressed, process when released
          keypressed&=~KP_KEYDOWN; // don't precess this key
          return; // don't even check as prefix key
        } else
        {
		      classicflags&=~F_FKEYPRESSED;  // f key was released, process as if pressed
          lastcolumncode=0x21; // cause keyreleased event
        }
    }
    flags&=~F_REMOTEKEY;

// blank dispay for better user response in fast mode
    if(DisplayNr!=DISPCALCULATOR || keyflags & KF_SUPRESSKEY)
    {
	    flags3|=F_DISPLAYOFF;  Wait8ms(3);	flags3&=~F_DISPLAYOFF;
    }

// ab hier Auswertung einer gltigen Taste 

    if(!(act_flags & AF_RUNMODE))
      flags2|=F_WAITKEY2ROM;  // calls ShowMnemonic 

    if(DisplayNr==DISPTEMP)
    {
      if(keyindex==PLUSKEY)
        TempOffset++;
      if(keyindex==MINUSKEY)
        TempOffset--;
      if(keyindex==CHSKEY)
        flags3^=F_FAHRENHEIT;
#if 0
      if(keyindex==0)
      {
        TempOffset=0;
        TempSum=TempADCValue<<4;
      }
#endif
      ShowTemperature();
    }

#ifdef HP55
    if(HPType==0x55)
    {
			if(prefixflags & PF_FKEY && keycode==0014)  // f g toggles time mode
				classicflags^= F_TIMER;
			prefixflags&=~PF_FKEY;
			if(act_flags & AF_RUNMODE && keycode==0016) // F_KEY
				prefixflags|=PF_FKEY;
    }
#endif

#if 0 // ausgeklammert um Programspace zu sparen
  if(TypeFlags & TF_SPICE)
  {
		  flags1&=~F_LOWBATT;
		  if(user_flags & UF_BATTCHECK && Voltage<LOWVOLTAGE)
			  flags1|=F_LOWBATT; // set Battery Low to invert decimals
  }
#endif
			
#ifdef HP01
      if(HPType==0x01)
      {
        SleepCnt=keycode==51 ? 3 : 7;  // show time for 3 seconds, else 7 seconds
      	hp01flags&=~F_SLEEP; // end sleep mode, start execution 
        if(keyindex==CHSKEY || keyindex==STOKEY)  // CHS will simulate f -  ( +/-), STO simulated f M
          act_s|=1<<4;  // set prefix bit, keycode is already set
      } else if(hp01flags & F_BLINK) // if Alarm triggered in HP-25 mode
      {
         hp01flags&=~F_BLINK;
         act_flags|=AF_DISPLAY_ON;
      }
#endif

#ifdef EXTENDEDFEATURES
      DoKeys();
#endif	
		}
		
// send keycode to act if key is pressed, must be called every instruction cycle

		if(keyflags & KF_ALLOWKEY) // end key suppress
		{
			keyflags&=~(KF_ALLOWKEY|KF_SUPRESSKEY|KF_FUNCTION|KF_STOPRGM|KF_RCLPRGM);
      keypressed&=~KP_KEYDOWN;
			
			if(keyflags & KF_CLEARPREFIX)
			{
         ClearPrefix();
			}
			else if(keyflags & KF_GTOSTEP)
			{
				keyflags &=~KF_GTOSTEP;
				act_press_key(GetIndexKeyCode(FKEY)); // prefix f replaces GTO prefix, which is not directly clearable by ENTER
				flags2|=F_PRESSENTER;
				flags2|=F_WAITKEY2ROM;
			}
      flags2&=~F_DOUBLEPREFIX;
		}

		if(keyflags & KF_SHOWPROGRAM) // shows program and single step, must be called every cycle because SST key is hold down half the time
		{
      if(HPType==0x34)
      {
	  		if(RepeatCnt==RepeatTime/2-5)
 		  		act_press_key(GetIndexKeyCode(HKEY));
      }

// key will be pressed in second half of RepeatTime
			if(RepeatCnt>=RepeatTime/2)
      {
#ifdef MNEMONICS
        if(RepeatCnt==RepeatTime/2+2) // +1 funktioniert nicht
          ShowMnemonic();
#endif
 				act_press_key(key);
      } else  if(act_flags & AF_RUNMODE) // im RUN mode wenn simulierte Taste losgelassen keine Mnemonic Anzeige
        flags&=~F_SHOWMNEMONIC;
		}

		if(keypressed & KP_KEYDOWN && !(keyflags & KF_SUPRESSKEY) && DisplayNr==DISPCALCULATOR) // any key hold down ? keys will not be given to ACT if stopwatch etc is enabled
		{
// if KF_REPEATKEY set then, press and relase button alternating every RepeatTime, press when RepeatCnt<RepeatTime/2 

// in der zweiten Hlfte der RepeatTime wird Taste nicht gedrckt

		  if(keyflags & KF_REPEATKEY)
      {
         if((keyindex==XYKEY && TimerCnt>=RepeatTime+RepeatDelayTime-1) || (keyindex==ROLLKEY && TimerCnt>=3*RepeatTime+RepeatDelayTime-1)) // stop rolling after first key and repeats
					keyflags&=~KF_REPEATKEY;


#ifdef MNEMONICS
         if((keyindex==sstkey || keyindex==bstkey) && RepeatCnt==1)
           ShowMnemonic();
#endif
			  if(RepeatCnt<=RepeatTime/2) // abwechlsend Taste gedrckt und losgelassen, erster Tastendruck lang, dann mit Wiederholrate
          goto L10;

        act_flags&=~AF_KEYDOWN; // for Classic

        if(HPType==0x34)
        {
				  if((keyindex==sstkey || keyindex==bstkey) && (RepeatCnt==RepeatTime-5)) // h Key 40 ms vor RCL key drcken
					  act_press_key(GetIndexKeyCode(HKEY)); // h SST //					act_press_key(HPKey[keyindex==RCLKEY ? HKEY : FKEY]); // h SST or f Roll
        }
#ifdef RIGHTSHIFT
				if(user_flags & (UF_RIGHTADJUST|UF_SYMBOLDISPLAY))  // zeige PI E auch bei XY und ROLL whrend repeat
					flags2|=F_WAITKEY2ROM;
#endif
      } else
      {
L10: 		 act_press_key(keycode); // must be called every cycle if key is pressed
      }
		}

    if(keyflags & KF_REPEATKEY)
    {
       LongPressCnt=0; // whrend repeat kein SwitchCalculator und kein Tracecode anzeigen
       KeyDownCnt=0;
    }

		if(keypressed & KP_RELEASED)  // key released ?
		{
			keypressed&=~KP_RELEASED; // key is processed
      keypressed&=~KP_KEYDOWN;
			keyflags&=~KF_REPEATKEY; // stop repeating
      act_release_key(); // necessary for HP01
      TraceCode=0;

      if(flags1 & F_SHOWKEY)  // Mnemonic shown ?
      {
        flags1&=~F_SHOWKEY;
        if(DisplayNr)
          ShowDisplayNr();
        else
          EnterCalculator(); 
      }

      if(HPType==0x38) // HP38 uses key for switching between RUN/PRGM
        if(act_s & 1<<11) act_flags&=~AF_RUNMODE; else act_flags|=AF_RUNMODE;

			if(classicflags & F_FKEYDOWN)
			{
        classicflags &=~F_FKEYDOWN;
				classicflags &=~F_FKEYPRESSED;
			}

      if(act_flags & AF_RUNMODE)
        flags&=~F_SHOWMNEMONIC; // beende Mnemonic Anzeige im Run Modus wenn Taste SST BST RS losgelassen
		}
}

// manage slide switch and battery low indicator for different calculators, must be called every instruction cycle 
inline void DoSlideSwitch()
{
  if(HPType==0x55)
  {
		if(classicflags & F_TIMER) // TIMER mode active?
			act_s |= 1<<11;  // s11  must be set in TIMER mode
		else if(!(flags & F_SLIDERRIGHT)) // s3 must be set in prgm mode
			act_s |= 1<<3;  // bit must be set for each instruction
  }
  else if(!(TypeFlags & TF_CLASSIC) && HPType!=0x67)
  {
// HP-27 and HP-31E/32E HP01 doesnt have PRGM/RUN switch, dont set s3, which is used for other purpose
  if(TypeFlags & TF_SLIDER) // HPType!=0x27 && HPType!=0x31 && HPType!=0x32)
  {
	 if(flags & F_SLIDERRIGHT)   // non programmables are always in RUN mode, switch has different meanings
			act_s |= 1<<3;  // bit must be set for each instruction
  }

  if(!(user_flags & UF_BATTCHECK) || Voltage>=LOWVOLTAGE) // if below, minimum 1.8 Volt is allowed for PIC and PCF2127 but doesn't work below 2.2V
	  act_s|=1<<5; // Battery OK ? must be set for each instruction
  }
}

#ifdef PCF2127
inline void DoInterrupt()
{
uint8_t i,buf[2];

    RTCRead(0,buf,2); // Read RTC interrupt flags
    i=buf[1] & 0x7;  // 0x07     // clear RTC interrupt flags MSF,AF, leave TSIE AIE CDTIE bit unchanged
    RTCWrite(1,&i,1);     // write RTC interrupt flags MSF,AF etc.

    if(buf[1] & 0x80) // MSF minute or second interrupt flag set?
    {
      flags4|=F_MSINT;
      if(!(flags & F_TIMEENTRY))
        RTCReadDate();  // read clock for remote read

      if(AlarmCnt==0 && DisplayNr>=DISPDATE && DisplayNr<=DISPDAY && !(flags & F_TIMEENTRY)) // dont read date if date entry is active 0x10 or Alarm is displayed
        ShowDisplayNr();
    }

    if(buf[1] & 0x10) // AIF alarm flag set ?
		{
		  if(hp01flags & F_ALARMACTIV) // alarm flag set and alarm active ?
			{
        if(!(flags4 & F_ALARMDAILY))
				  hp01flags &=~F_ALARMACTIV; // clear alarm
				AlarmCnt=ALARMTIME;  // ring alarm 10 beeps
        ReadText(10); // ALARM
        flags3|=F_INFOTEXT; // Text wird quittiert mit nchstem Tastendruck
			}
		}

// ON/OFF switch is connected to GND when ON, which sets TSF1 and TSF2
    if(buf[1] & 0x20) // Time Stamp TSF2 Flag set when /TS goes Low Power switch moved to OFF position
    {
        EndInfoText(); // Clear "OFF" text, reset F_OFF
//      ReadText(14); // "ON",2);  // switch ON
//      flags3|=F_INFOTEXT; // Text wird quittiert mit nchstem Tastendruck
    }
// when switched to OFF only TSF1 is set
    if(buf[0] & 0x10) // Time Stamp TSF1 Flag set when /TS goes to half Power level Power switch moved to ON or OFF position
    {
        i=1;       // clear TSF1, set SI
        RTCWrite(0,&i,1); 
        if(!(buf[1] & 0x20))
        {
          SleepCnt=2; // prohibit Sleep in the next seconds before OFF is processed

//          DisplayNr=DISPCALCULATOR;  // in case GPS Menu was active
//          if(keyflags&KF_SUPRESSKEY)
//         	  keyflags|=KF_CLEARPREFIX|KF_ALLOWKEY;

          ReadText(15); // "0FF",2);
          flags3|=F_INFOTEXT; // Text wird quittiert mit nchstem Tastendruck
          flags4|=F_OFF;
        }
    }
}
#endif

// set act_sw from t, t containes stopwatch time in 1/100 seconds
void set_act_sw(uint32_t t)
{
  uint8_t k,i;
  for (i = 0; i < 8; i++) // set 8 digits HH:MM:SS.HH
  {
    if (i == 3 || i == 5)
      k = 6;
    else
      k = 10;
    act_sw[i] = t % k;
    t = t / k;
  }
}

void CountStopwatch()
{
  if(TMR1IF) // Overflow every 512 seconds (8 Minutes 32 seconds)
  {
    TMR1IF=0;
    if(!(hp01flags & F_SWDEC)) // counting upwards ?
      SWStartTime++;       // add 512 seconds
    else
    {
      if(SWStartTime==0)  // countdown zero reached
      {
        hp01flags &= ~F_SWDEC; // count upwards after reaching zero
        AlarmCnt=ALARMTIME;  // start Alarm
      } else
        SWStartTime--; // subtract 512 seconds
    } 
  }
}

// wird wiederholt aufgerufen wenn die Stopuhr luft und bei Stop oder key pressed
void ReadStopwatch()
{
int32_t t;
uint16_t T1Value;

//   if(DisplayNr==DISPSTOPWATCH && !(flags & F_TIMEENTRY)) // stopwatch displayed ?
   {
      T1Value=TMR1H<<8 | TMR1L; 

#if 0
// because the Timer is counting with 128 Hz overflow occurs every 512 seconds and TMRH and TMRL are read within 1 us,
// it is very unlikely that TMRL overflows during read. If overflow occurs, only wrong display for 1/100 sec visible

      if(TMR1L<(uint8_t)T1Value // check Overflow ?
        T1Value=TMR1H<<8 | TMR1L; // read again
#endif

      if(hp01flags & F_SWDEC)
        T1Value=-T1Value;   // in decrement mode T1Value is negative

#if 0
      t=(T1Value<<4);  // braucht deutlich mehr Programmschritte dafr wahrscheinlich schneller
      t+=(T1Value<<3);
      t=t+T1Value>>5;
#endif

      t=(T1Value*25L)>>5;  // *100/128 (25/32),  always positive value 0 - 51199 1/100 s
      t += SWStartTime*51200L; // add StartTime

      set_act_sw(t);
      ShowStopwatch();
   } 
}

void NextProgram()
{
uint8_t n;
          n=GetMaxPrograms();

          if(keyindex==CHSKEY)
             flags3^=F_SHOWPROGRAMNR;
			    else if(keyindex==sstkey)
				    ProgramNr+=10;
			    else if(keyindex==bstkey)
				    ProgramNr-=10;
			    else if(keyindex==PLUSKEY)
				    ProgramNr++;
			    else if(keyindex==MINUSKEY)
				    ProgramNr--;
          else
            return;
          if(ProgramNr>=240) ProgramNr=0; // unsigned underflow
          if(ProgramNr>=n) ProgramNr=n-1; // max 165
          ShowProgramName();
}

// wird von DoTimer alle 8 ms aufgerufen
void DoUserStringEntry()
{
	uint8_t c;
  char *pt=(char *)rombuf;

    if(keypressed & KP_KEYDOWN) // key hold down ?
    {
  		if((EntryTimerCnt & 31)==0)  // Next char alle 32*8ms
      {
        EntryTimerCnt=0; // zurcksetzen damit Cursor angezeigt wird

        if(DisplayNr==DISPDIRECTORY)
        {
          NextProgram();
          goto L2;
        }
	      c=pt[ScrollIndex+StringIndex];
        if(hpcode==0x51) // + ein character vor im Alphabet
 	      {
          if(c<':' || ++c>=MAXALPHA) // : is lowest character
		        c=':';
         	goto L3;
        }
				if(hpcode==0x41) // - ein character zurck im Alphabet
    	  {
      	  if(c==0 || --c<':')
	  	      c=MAXALPHA-1;
L3:       pt[ScrollIndex+StringIndex]=c;
          EntryTimerCnt=32; // show actual Text instead of cursor
        } 
    		if(hpcode==0x11 /*|| hpcode==0x61*/) // SST  Cursor nach rechts
        {
          if(ScrollIndex==MAXDIGITS-1)
          {
            if(!(flags1& F_ENTERPRNAME) && StringIndex<USERSTRINGLEN-MAXDIGITS)
			        StringIndex++;
          }
          else
				    ScrollIndex++;
        }
			  if(hpcode==0x12 /*|| hpcode==0x71*/) // BST Cursor nach links
        {
          if(ScrollIndex==0)
          {
            if(StringIndex>0)
	            StringIndex--;
          }
          else
				    ScrollIndex--;
        }
      }
    }

    if(DisplayNr==DISPUSERSTRING) // if necessary otherwise blinking cursor will be displayed
      if((EntryTimerCnt & 31)==0)  // show text eversy 1/2 second for cursor blinking
        ShowUserStringEntry();

L2:
	  if(++EntryTimerCnt>=64)  // Cursor blinkt alle 64*8ms = 0.5 s 
		  EntryTimerCnt=0;
}

void SwitchCalculator()
{
    uint8_t n=keyindex;
    uint8_t Type;

    if(n<NROFCALCULATORS)
    {
       if(Type=SST25SetRomAddr(n)) // dont set HPType here, bacause WriteContinousMemopry use the old type
       {
#ifdef CONTMEMORY
         ReadText(24); // CALCULATOR
         WriteContinuousMemory(HPTypeNr); // save registers and context
	       IncrementWriteCycles();
#endif
         DefaultTypeNr=n; // save last used calculator in Flash
         HPTypeNr=n;
         HPType=Type;
         SetupHPType();
#ifdef CONTMEMORY
         ReadContinuousMemory(HPTypeNr);
         if(act_pc==0xffff)  // Flash processor context uninitialisiert
           act_reset();
#else
         act_reset();
#endif
         systemflags=0;
         DebugCommand=0;
         EnterCalculator();
         flags|=F_SHOWINFO;
         ShowLogo();
  			 flags3|=F_INFOTEXT; // end Logo display after some seconds
				 keyflags|=KF_SHOWINFO; // end Logo display after some seconds
         TimerCnt=0;
        }
    }
}

// If no program is running check PRGM/RUN Slider for positive or negative edge. If changed set/reset SLIDERRIGHT and call SwitchPrgmRun
inline void CheckSlideSwitch()
{
  if(systemflags & SF_SLIDERCHANGED) // switch PRGM/RUN by Remote command
  {
    systemflags&=~SF_SLIDERCHANGED;
    SwitchPrgmRun();
  }

  if(!(flags & F_PRGMRUN) && PRGMRUN || (flags & F_PRGMRUN) && !PRGMRUN)  // RUN/PRGM Switch changed ?
  {
// switch calculator model
     {
       if((!(TypeFlags & TF_CLASSIC) || HPType==0x55) && !(act_s & RunBit))  // wait until program execution is stopped
		   {
			    flags^=F_PRGMRUN;      // store new state of physical slider
				  flags&=~F_SLIDERRIGHT;
				  if(flags & F_PRGMRUN)  // non programmables are always in RUN mode, switch has different meanings
			  	  flags|=F_SLIDERRIGHT;

  			  SwitchPrgmRun();
	      }
      }
#ifdef HP55
			classicflags&=~F_TIMER; // end HP-55 timer mode if slider switch changed
#endif
  } 
}

// must be called at least every 8 ms
inline void DoTimer()
{
uint8_t i,a;

  if(DisplayNr==DISPUSERSTRING || DisplayNr==DISPDIRECTORY)
    DoUserStringEntry();

#ifdef SHOWROM
  if(DisplayNr==DISPROM)
  {
		if(EntryTimerCnt && --EntryTimerCnt==0) // autorepeat
		{
      if(keypressed & KP_KEYDOWN)
      {
			  EntryTimerCnt=5;  // Wiederholrate 40 ms
        DoShowROM();
        ShowDisplayNr();
      }
    }  
  }
#endif
    
    if(DisplayNr==DISPTEXT)
    {
      if(TickerSpeed<150) // kein Stillstand?
      {
      	if(++TickerCnt>=TickerSpeed)  // ticker speed
      	{
        	TickerCnt=0;
        	a=GetTickerText(TickerIndex);
       		if(a==TICKEREOT)            // Laufschrift auf nchsten Buchstaben weiterschalten
       			TickerIndex=0;
       		else
         		TickerIndex++;
        	if(!(flags3 & F_INFOTEXT))
	        	ShowTicker();
      	}
      }
    }

    if(hp01flags & F_SWSTARTED) // whrend Stopuhr luft nur alle 1/10 Sekunde anzeigen um Rechenzeit zu sparen
    {
      CountStopwatch(); // check overflow and alarm even if stopwatch is not displayed

// Die Berechnung der Stopuhrzeit dauert relativ lange und wird deshalb nur ausgefhrt wenn die Anzeige aktiv ist

      if(DisplayNr==DISPSTOPWATCH && !(flags3 & F_INFOTEXT) && !(flags & F_TIMEENTRY))
      {
        if((TimerCnt & 7)==0)
          ReadStopwatch();  // prft auch Alarm
        else
          act_reg[3]=(char)TimerCnt & 7;  // simuliere schnell laufende Stopuhr im 1/100 digit
      }
    }

    if(BeepCnt && --BeepCnt==0)
      EndSoundBeep();

  	if(AlarmCnt) // Alarm active ?
    {
      if(SleepCnt==1)  // don't sleep while alarm active
        SleepCnt++;

      if(++AlarmTimerCnt>=40)
      {
         AlarmTimerCnt=0;
         AlarmCnt--;
         if(AlarmCnt & 1)
           SoundBeep();
         else
           EndSoundBeep();
         if(AlarmCnt==0)
         {
           if(DisplayNr==DISPCALCULATOR)  // Alarm Textanzeige beenden
             flags&=~F_SHOWINFO;
           SleepCnt=1; // switch off if no key was pressed
           flags4|=F_OFF;
         }
      }
    }

// Spannung und Temperatur wird ca. alle 2 Sekunden gemessen 256*8 ms

  	if((char)TimerCnt<=3)
    {
#ifdef SHOWTEMP
			if((char)TimerCnt==0)
   			ADCON0=30<<2 | 1; // select channel 30 Temperature Value
      if((char)TimerCnt==1)
        GO_nDONE=1; // start conversion of temperature min 200us after channel select
      if((char)TimerCnt==2)
      {
        TempADCValue = ADRESH<<8 | ADRESL; // TempSum-=TempSum>>4; TempSum+=TempADCValue;
        TempADCValue = 1023-TempADCValue; /*(TempSum>>4)*/; // Vt in digits von Vdd 1023=Vdd,  458 bei Vdd 2.5V
        Temperature=(uint32_t)TempADCValue*2048/RefADCValue *128/338;  // gemessene Temperatur in C ist konstant bei unterschiedlicher Vdd ergibt ca 160 bei 20C
        Temperature=-Temperature+TempOffset;
      }

#endif
#ifdef SHOWVOLTAGE
      if((char)TimerCnt==2)
      {
 			  ADCON0=31<<2 | 1; // select channel 31 Fixed Voltage 4.096 to measure VDD
        GO_nDONE=1; // start conversion temperature/voltage
      }
      if((char)TimerCnt==3)
      {
        RefADCValue = ADRESH<<8 | ADRESL;
        // 16-bit Berechnung von Vdd aus RefADCValue, 1024=2,048V  Vdd=2048*1024/RefADCValue in mV
        Voltage=(uint16_t)52429/(RefADCValue>>2); // Ergebnis in 1/100 Volt, 52429 = 16-bit Nherung 2048*1024/10/4, Zhler und Nenner/4 vermeidet 16-bit Overflow
      }
#endif
    }

#ifdef STOPWATCH
    if(user_flags1 & UF_RUNTIME)
    {
       if(!(hp01flags & F_SWSTARTED) && act_s & RunBit) // user program is running ?
       {
         flags1|=F_TIMESTARTED;
         StartStopwatch();
       }
       if((flags1 & F_TIMESTARTED) && !(act_s & RunBit)) // user program is not longer running ?
       {
         flags1&=~F_TIMESTARTED;
         StopStopwatch();
       }
    }
#endif

    if(PauseCnt) PauseCnt--;

#ifdef SHOWKEY
// nach 1 sek key text einblenden

    if(TraceCode && keypressed & KP_KEYDOWN &&  ++KeyDownCnt==100)
    {
      if(DisplayNr==DISPUSERFLAGS)
      {
         ReadText(TraceCode); // show User Flags text
      } else
      {
#ifdef MNEMONICS
        ClearInfo();
        SetMnemonic(TraceCode,0x4000+MnemonicAddr); // show program step mnemonic
#endif
      }
      flags1|=F_SHOWKEY;
      TraceCode=0;
      LongPressCnt=0; // ab jetzt Zeit fr SwitchCalculator starten
    }
#endif
    	
		if(++SecondCnt>=122) // 122*8,125 ms 1 Sekunde auf 0,5 Promille genau
		{
			SecondCnt=0;

			IncrementOperatingTime();  // auch wenn Echtzeituhr, dann hier incrmentieren

      if(DisplayNr==DISPVOLTAGE || DisplayNr==DISPTEMP || DisplayNr==DISPOPERATINGHOURS)
        ShowDisplayNr();

#ifdef PCF2127
      if(!(flags4 & F_RTCINIT))
      {
        flags4|=F_RTCINIT;
    	  RTCInit(); // Initialize RTC after CLKOUT is present
      }
#endif

      if(keypressed & KP_KEYDOWN /*&& !(keyflags & KF_REPEATKEY)*/ && !(flags2 & F_PREFIX) && ++LongPressCnt==3)
      {
         if(keyindex==DIVKEY) // goto Sleep if DIV key is pressed long
         {
            SleepCnt=3;
            goto L1;
         }
         else if(keyindex==1)
         {
            DisplayNr=DISPLOGO;
 		        ShowLogo();
         }
      }

			if(user_flags & UF_SLEEPMODE && !ENABLEGPS)
			{
        if(!(act_s & RunBit)) // user program is not running ?
        {
				  if(SleepCnt==3) // drei Sekunden vor GotoSleep SLEEP anzeigen, damit noch abgebrochen werden kann
          {
L1:
            ReadText(9); // SLEEP
            flags3|=F_INFOTEXT; // Text wird quittiert mit nchstem Tastendruck
            flags4|=F_OFF; // aktion kann noch mit Taste abgebrochen werden
          }
        }
			}

		  if(SleepCnt && --SleepCnt==0) // wird decrementiert um automatisch auszuschalten
      {
        if(flags4 & F_OFF) // ist nur gesetzt wenn UF_SLEEPMODE aktiv oder manuell ausgeschaltet
        {
           Readrombuf(FLASHVARPAGE,FLASHVARADDR); // kopiere 64 byte
           for(i=0;i<FLASHVARSIZE;i++)
             rombuf8[i]=FlashVars[i];  // aktualisiere
           SST25WriteToPage(FLASHVARPAGE,FLASHVARADDR,rombuf8); // avoid Text "Flash Write"
//         Writerombuf(FLASHVARPAGE,FLASHVARADDR);
           GotoSleep();
         } else
           SleepCnt=SLEEPTIME;
      }

		}

		if(keyflags & KF_SHOWREGISTER)  // Registeranzeige
		{
				i=TimerCnt>>8;  // Registernr 0-7
				if((char)TimerCnt==0)
				{
					ClearInfo();
					act_reg[WSIZE-1-11]=i; // rightmost digig like HP-67
					act_press_key(GetIndexKeyCode(RCLKEY));  // RCL
				}
				if((char)TimerCnt==32)
				{
					flags&=~F_SHOWINFO; // end display registernr
					act_press_key(GetIndexKeyCode(i));
#if 0
					register_to_c(i);
					dest=act_n,src=act_c,reg_compress();
					act_press_key(GetIndexKeyCode(GKEY));
					flags2|=F_WAITKEY2ROM; // rightadjust
#endif
					if(i>=PrimaryRegisters-1) // stop after RCL 7 or RCL 9 (HP-29C/HP34C)
					{
						keyflags&=~KF_SHOWREGISTER;
						keyflags&=~KF_SUPRESSKEY;
					}
      }
		}
		else if(keyflags & KF_SHOWINFO)  // Mantisse Anzeige
		{
			if(TimerCnt & 2048/8 && !(keypressed & KP_KEYDOWN))
      {
				keyflags&=~KF_SHOWINFO;
        EndInfoText();
      }
		}

		else if(flags2 & F_PRESSENTER && !(flags2 & F_WAITKEY2ROM) && !(flags2 & F_WAITIDLE)) // GTO prefix muss mit f ENTER beendet werden
		{
			if(TimerCnt>=2) // less than 10 doesnt work HP-29 internal ROM
			{
				flags2&=~F_PRESSENTER;
				keyflags&=~KF_CLEARPREFIX;
				prefixflags&=~PF_FKEY;
				prefixflags&=~PF_GKEY;
				act_press_key(GetIndexKeyCode(ENTERKEY));
#ifdef RIGHTSHIFT
				if(user_flags & UF_RIGHTADJUST)
					flags2|=F_WAITKEY2ROM;  // for rightadjust
#endif
			}
		}
		else if(keyflags & KF_SHOWPROGRAM  && !(act_flags & AF_RUNMODE) && RepeatCnt==RepeatTime-1)  // Programmanzeige ende zeitlich nach dem SST Tastendruck
		{
      TriggerSleepTime(); // don't go slepp during showprogram
			i=GetProgramStep();
      if((HPType==0x34 && !(flags3 & F_FIRSTSTEP)) && i==0 || // HP-34C has dynamic Program Steps
			  (i==MaxProgramSteps || ProgramIsEmpty(i))) // stop at step 49/98/224 or when empty
			{
//        StopShowRegister();
				keyflags&=~KF_SHOWPROGRAM;
				keyflags&=~KF_SUPRESSKEY;
				keyflags|=KF_ALLOWKEY;
			}
			flags3&=~F_FIRSTSTEP;
#ifdef HP82240
			if(user_flags & UF_HP82240 && i>0 && keyindex!=BSTKEY) // nur vorwrts
			{
				SendHP82240ProgramStep(GetProgramCode(i),1); // print program step
#ifdef MNEMONICS
        ShowMnemonic(); // restores display Mnemonic, which was overwritten by PrinterMnemonic
#endif
			}
#endif
		}

		TimerCnt++;
    if(TimerCnt>=RepeatDelayTime) // DelayTime followed by RepeatTimes
    {
      if(++RepeatCnt>=RepeatTime)
          RepeatCnt=0;
    }
}

// After HPType is set set Variables specific to HPType  
void SetupHPType()
{
// set keyboard code tables

  DispHPType=HPType;
  DispHPTypeNr=HPTypeNr;
  PrimaryRegisters=8; // default HP-25
  WaitKeyReleasedAddr=-1; // 0 causes show Menmonic at reset
  MaxProgramSteps=0;
  RunBit=0;
  annuncmask=0;
	sstkey=SSTKEY;
	bstkey=BSTKEY;


 	flags&=~F_PROGRAMMABLE; // assume not programmable
  if(ProgramPages>0) // HPType==0x55 || HPType==0x65 || HPType==0x25 || HPType==0x67 || HPType==0x33 || HPType==0x34 || HPType==0x38)
    flags|=F_PROGRAMMABLE;

  if(HPType==0x21)
  {
	  annuncmask=1<<13; // bit 13 set in HP-21 if g pressed
    PrimaryRegisters=0;
  }
  if(HPType==0x25)
	{
		annuncmask=(1<<9|1<<10|1<<14); // bits 9,10,14 set in HP-25 when f g STO pressed
    RunBit=1<<1;
    MaxProgramSteps=49;
    WaitKeyReleasedAddr= 01462;
	}		
	if(HPType==0x22)
	{
		PrimaryRegisters=10;
    annuncmask=1<<4;
	}
	if(HPType==0x27)
	{
		PrimaryRegisters=10;
	  annuncmask=1<<14; // bit 14 set if any prefix f g pressed
 	}
  if(HPType==0x29) // || HPType==0x19)
  {
    bstkey=BSTKEY;  // doesnt have BST key, but use GSB instead
	  annuncmask=1<<13; // bit 13 set in HP-29 if any prefix pressed
    RunBit=1<<2;
	  flags|=F_PROGRAMMABLE;
    PrimaryRegisters=10;
    MaxProgramSteps=98;
    WaitKeyReleasedAddr= 0260;
  }
  if(HPType==0x31)
  {
	  annuncmask=1<<6; // bit 13 set in HP-21 if g pressed
  }
  else if(HPType==0x37)
  {
	  annuncmask=1<<13; // bit 13 set if f prefix pressed
  }
  else if(HPType==0x33)
  {
		annuncmask=1<<4; // f prefix = s4 g prefix = s4+6
    MaxProgramSteps=49;
    WaitKeyReleasedAddr=03202;
  }
  else if(HPType==0x34)
  {
  	sstkey=RCLKEY;
  	bstkey=STOKEY;
		PrimaryRegisters=10;
		annuncmask=1<<14 | 1<<7;
    MaxProgramSteps=210;
    WaitKeyReleasedAddr=012213;
    RunBit=1<<2;
  }
  if(HPType==0x67)
  {
    bstkey=-1;  // no BST key
		annuncmask=1<<13; // bit 13 set in HP-29 if any prefix pressed
    RunBit=1<<2;
    MaxProgramSteps=224;
    WaitKeyReleasedAddr=01106;
  }
}

// Init special functions register at start up
inline void Init()
{
	OSCCON=0x7A; // internal FOSC 7A=16 MHz  6A=4 MHz

	PORTA=0xFF;  // /CE High /TS High, rows 1-6 high, no rows selected
	PORTB=0x00;  // all inputs
	PORTC=0x00;  // no segments selected, BUZZER Low,
	PORTD=0x00;  // no segements selected
	PORTE=0x03;  // CE,CE2 high GPS low

	TRISA=TRISPORTA;
	TRISB=TRISPORTB;
	TRISC=TRISPORTC;
	TRISD=TRISPORTD;
	TRISE=TRISPORTE;

  ANSELA=0x00;  // Digital I/O // analog input after reset
  ANSELB=0x00;  // Digital I/O
  ANSELC=0x00;  // Digital I/O
  ANSELD=0x00;  // Digital I/O
  ANSELE=0x00;  // Digital I/O

	OPTION_REG=0x06; // Timer0 internal clock FOSC/4, Prescaler assigned to Timer, rate 1:128, pull ups enabled
	WPUB=0x3F; // column PB1-5 pullup, /INT pullup

	ADCON1=0xA0; // 0x50=FOSC/16 1 MHz 0x20=FOSC/32 500kHz AD clock -> 24us (44 allowed), bit7 right justified
// The fixed Voltage Reference is always enabled when HFINTOSC internal High Frequency Oscillator is used
  FVRCON=0xA2; // TSEN=0, Temperature sensor enabled, Low Range  Vout=Vdd-2xVt, FVR enabled 2x1.024V = 2.048V

  StartDisplay();

// Setup Timer 1 for external Stopwatch
  T1CON=0xB4;  // external 1024 Hz clock T1CKI PC0, Prescale 1:8 T1SYNC=1 for running in Sleep mode
  TMR1H=0;
  TMR1L=0;
  TMR1ON=0;

#ifdef SERIALPORT
	InitUART();
#ifdef NMEAMENU
  NMEAInit();
#endif	
#endif

  SST25Init(); // initialize SPI bus

// initialize variables

	if(!(PORTA & F_PRGMRUN))
    flags=F_PRGMRUN; // cause Switchprgmrun at program start

	SleepCnt=SLEEPTIME;
  TickerSpeed=50;
	Voltage=LOWVOLTAGE; // Voltage>=LOWVOLTAGE damit nach Einschalten kein Low Power angezeigt wird

  SST25Read(FLASHVARPAGE,FLASHVARADDR,FlashVars,FLASHVARSIZE); // read user_flags, 4 words OperatingHours 2 bytes OperatingTime 3 words WriteCycles, DefaultType
  if(DefaultTypeNr>=NROFCALCULATORS) // zur Sicherheit, falls FlashVars korrupt
  {
    memset(FlashVars,0,FLASHVARSIZE); // this clears operating time and write cycles
#ifdef SPICE
    DefaultTypeNr=7; // HP-33
#else
    DefaultTypeNr=2; // HP-25
#endif
  }
  HPTypeNr=DefaultTypeNr;  // 2=HP-25 3=HP-27

#ifdef HP21HW
  user_flags1&=~UF_HP21HW;
#endif

#ifdef XDEBUG
  HPType=0x55;
  TypeFlags|=TF_CLASSIC;
  ProgramPages=1;
  RegisterPages=1;
#else
  HPType=SST25SetRomAddr(HPTypeNr);
#endif
  SetupHPType();

// initialize Variables with absolute address, which are not cleared by C compiler

	PEIE=1; // erlaube peripheral interrupts (Timer, TXIF, RXIF)
	GIE=1; //  erlaube interrupts
}

// clear complete internal RAM memory banks
// this is more efficient than initializing by linker/compiler
// values!=0 have to be initialized after ClearMem()
inline void ClearMem()
{
  FSR0H=0;
  do
  {
    FSR0L=0x20; // clear RAM bank from 0x20 to 0x7f and 0xa0-0xff 2*96 bytes
    do
    {
      INDF0=0;
      if(++FSR0L==0x80) FSR0L=0xA0; // skip 0x80-0x9F special function registers
    } while(FSR0L!=0);
  } while(++FSR0H<12); // clear all 12 banks of PIC16LF1519
}

// main program entry and endless loop, ACT emulator
int main()
{
uint8_t i;

  ClearMem();

	Init(); // initialize special function registers and variables

	act_reset();

  ShowLogoInfoText();

#ifndef XDEBUG
// Emergency reset falls Flash Context defekt, alle Tasten DIV 0 , RS  gedrckt beim Einschalten
  PORTA=0x43; // select Row 7 
  PORTA=0x43; // select Row 7 needs one nop time before read PORTB
  i=PORTB & 0x3E;  // no key=3E DIV=3C  0=3A DOT=36 R/S=2E ALL=20
  if(i != 0x3E) // any key pressed in row 7 ?
  {
#ifdef HP21HW
    if(i==0x3A) // Dot key pressed for testing LP circuit on HP-21 hardware
      user_flags1|=UF_HP21HW;
    else
#endif
    {
    	ReadText(26); // RESET
    	SST25Erase(CONTEXTPAGE,ACTSTATEADDR); // clear 4k ProcessorState Area
    	SST25Erase(CONTEXTPAGE,CONTMEMADDR); // clear 4k Continuous Memory Area
    	HPTypeNr=2; // HP-25
    	HPType=SST25SetRomAddr(255); // 255=set default calculator HP-25 without using FlashInfo
    	flags3&=~F_INFOTEXT; // falls Logo angezeigt wird, sonst lscht die hier gedrckte Taste den Text
	  	keyflags|=KF_SHOWINFO; // end text display after some seconds
    	TimerCnt=0;
    }
  }
#endif

//  fast main loop every instruction cycle

	while(1)
	{
// Abfrage TMR0IF vor DoTimer ist schneller
  	if(TMR0IF) // every 8,1920 ms 16/4 Mhz/256/128
	  {
		  TMR0IF=0;

#ifdef EXTENDEDFEATURES
		  DoTimer();
#endif

#ifdef SERIALCOMMAND
  		if(!ENABLEGPS && CheckSer())
	  		DoSerialCommand();
#endif
    } 

#ifdef PCF2127
// Abfrage INTF vor Aufruf DoInterrupt ist schneller, Abfrage INTF ist nicht zuverlssig, deshalb wird INT Pegel abgefragt
  	if(INT==0)  // SI Seconds INT is generated 70 ms before 1Hz CLKOUT goes low, may vary??  
			DoInterrupt();
#endif

#ifdef NMEAMENU
		if(ENABLEGPS && flags1 & F_TICK && !(flags4 & F_OFF))  // ca. 1 ms
    {
      flags1&=~F_TICK;
			if(NMEADo()) // call every ms if NMEA is active, returns true if data received
      {
        if(DisplayNr==DISPNMEA) // is NMEA display active?
            NMEAShowData(); // display new data after position is received, every second
      }
    }
#endif

#ifdef HP55
		if(classicflags & F_TIMER) // slow down execution in timer mode
		{
       // HP-55 Timer soll mit Stopuhr synchronisiert werden
		}
#endif

#ifdef ICDEBUG
// In circuit debugger
// 0x00 Run command 
// 0x01 Reset command
// 0x02 SingleStep command 
// 0x7f Stop command

  if(!(systemflags & SF_BPTRIGGERED)) // dont execute if breakpoint is triggered, must be reset by Host
  {
    if(DebugCommand!=0x7f) // 0=Run 1=Reset 2=Single Step
    {
      if(DebugCommand==1)    // reset command
        act_reset();      
      else                   // don't execute cycle after reset
        act_execute_cycle(); // run emulator

      if(DebugCommand!=0)    // Stop or SingleStep or Reset
        DebugCommand=0x7f;

 	    opcode = Getopcode();

    	if(systemflags & SF_BPENABLE && opcodeaddr==act_breakpoint || systemflags & SF_CODEBPENABLE && opcode==codebreakpoint )
        systemflags|=SF_BPTRIGGERED;
    }
  }
#else
     act_execute_cycle(); // run emulator
#endif

    if(flags1 & F_DISPTICK)
    {
      flags1&=~F_DISPTICK;
      DoDisplay();
      TMR2Cnt++;
    }

	  CheckKeyPress(); // check keys pressed or released
		DoSlideSwitch(); // set status bits on slider position
    CheckSlideSwitch(); // teste ob Zustand des Sliders sich gendert hat
	}
}
