#include <stdbool.h>
#include <stdint.h>
#include <stdlib.h>
#include "xc.h"
#include "hp25lp.h"
#include "act.h"
#include "sst25pf040.h"
#include "opcode.h"
#include "opfcn.h"

// ACT register
reg_t act_a;   // display register
reg_t act_b;   // decimal point, minus sign register
reg_t act_c;   // user stack x
creg_t act_y;  // user stack y
creg_t act_z;  // user stack z
creg_t act_t;  // user stack t
creg_t act_m;  // store register
creg_t act_n;  // store register
//uint8_t act_switch;

reg_t act_reg; // general register
reg_t act_dsp; // display register
reg_t act_cl;  // clock register
reg_t act_sw;  // stopwatch register
reg_t act_al;  // alarmregister

// variables used in assembler routines are located in Bank 0
uint8_t hp01flags;
uint8_t varbank0; // variable located in ram bank 0

uint8_t *src;
uint8_t srch;
uint8_t *dest;
uint8_t desth;
uint8_t *src2;
uint8_t src2h;
uint8_t first;
uint8_t last;
uint8_t s2;
uint8_t op6;
uint16_t opcode;
uint8_t ProgramNr;
uint8_t act_flags;
uint8_t crc_flags;  // card reader chip flags
uint8_t base;
uint16_t act_pc; // program counter
uint8_t act_del_rom; // delayed rom instruction
uint8_t act_sp; // stack pointer
uint8_t act_p;       // 4-bit digit counter
inst_state_t act_inst_state; // instrucstion state
uint16_t act_s;  // status register
uint8_t act_f; // flag register
uint8_t act_key_buf; // key buffer register
uint8_t act_ram_addr;  /* selected RAM address */
uint8_t act_ram_size;
uint16_t act_stack[STACK_SIZE]; // stack

// allocate 8*56 byte for ACT_RAM
//#define LINEARRAM
#ifdef LINEARRAM
uint8_t act_ram[448] @ 0x20A0; // reserve 448 bytes for ACT RAM, compiler can't allocate
#else
uint8_t act_bank2[80]; 
uint8_t act_bank3[80]; 
uint8_t act_bank4[80]; 
uint8_t act_bank5[80]; 
uint8_t act_bank6[80]; 
uint8_t act_bank7[48];
#endif

uint16_t instructioncnt;
uint16_t act_breakpoint; // debugging breakpoint
uint16_t codebreakpoint; // debugging breakpoint

const uint8_t p_set_map[16] = { 14,  4,  7,  8, 11,  2, 10, 12,  1,  3, 13,  6,  0,  9,  5, 14 };
const uint8_t p_test_map[16] = {  4,  8, 12,  2,  9,  1,  6,  3,  1, 13,  5,  0, 11, 10,  7,  4 };
const uint8_t p_set_map01[16]  = { 255, 11, 8, 0, 5, 255, 9,   1, 4, 2, 255, 3, 7, 6, 10, 255 };         // 255 is undefined opcode
const uint8_t p_test_map01[16] = {   1, 11, 8, 0, 5, 255, 9, 255, 4, 2, 255, 3, 7, 6, 10, 255 };

#ifndef USEASM
//#ifndef FLASHFCNTABLE

#ifdef INLINEASM1 // cannot compile "xc8 internal error"
void op_nop ()
{
#asm
_op_unknown:
_op_display_reset_twf:
_op_rom_addr_to_buf:
_op_crc_test_motor_on:
_op_crc_test_card_in:
_op_crc_write_prot:
_op_crc_motor_on:
_op_crc_motor_off:
_op_crc_test_prot:
#endasm
}
#else
void op_nop () { }
void op_unknown () { }
void op_display_reset_twf () { }
void op_rom_addr_to_buf() { }  /* I don't know what this instruction is supposed to do! */
void op_crc_test_motor_on () { }
void op_crc_test_card_in () { } // liefert s3 =1 wenn card detected
void op_crc_write_prot () { }
void op_crc_motor_on () { }
void op_crc_motor_off () { }
void op_crc_test_prot () { }
#endif

void op_crc_set_f0(){	crc_flags |=1<<0;}
void op_crc_set_f1(){	crc_flags |=1<<1;}  // start Pause
void op_crc_set_f2(){	crc_flags |=1<<2;}
void op_crc_set_f3(){	crc_flags |=1<<3;}
void op_crc_set_f4(){	crc_flags |=1<<4;}

void op_crc_clear_f0()
{
  if(crc_flags & 1)
  	act_s|=1<<3;
	crc_flags &=~1;
}

void op_crc_clear_f1() // test Pause pending
{
  if(crc_flags & 2)
  	act_s|=1<<3;
	crc_flags &=~2;
}

void op_crc_test_f1()
{
  if(!(act_flags & AF_RUNMODE))
    act_s|=1<<3;
  crc_flags &=~2;
}


void crc_test_f(uint8_t mask)
{
  crc_flags &= ~0x80;  if(act_s & 1<<3)  crc_flags|=0x80; // set delayed flag if s3 is set
  act_s&=~(1<<3);  // reset s3
  if(crc_flags & mask) 
     act_s |= 1<<3;
  crc_flags &= ~mask;   if(crc_flags & 0x80)  crc_flags |= mask;  // set f2 if delayed flag is set
}

void op_crc_test_f2()
{
  crc_test_f(1<<2);
}

void op_crc_test_f3()
{
  crc_test_f(1<<3);
}

void op_crc_test_f4()
{
  crc_test_f(1<<4);
}

void op_bank_switch()
{
  act_flags ^= AF_BANK;
}

void op_set_s()
{
	act_s |= (1 << op6);
}

void op_clr_s ()
{
	act_s &= ~(1 << op6);
}

void op_binary()
{
	base = 16;
}

void op_decimal()
{
	base = 10;
}

void op_clear_reg ()
{
  dest=(uint8_t *)act_a;
  first=0; last=3*WSIZE+3*WSIZE/2-1; // clear a,b,c,y,z,t   // woodstock does not clear act_m, f STK will keep display format in act_m
  if(TypeFlags & TF_CLASSIC)        
    last+=2*WSIZE/2; // clear also act_m = (act_n) = 0 only classic
  reg_zero();

#if 0
  uint8_t i;
  for (i = 0; i < WSIZE; i++)
    act_a [i] = act_b [i] = act_c [i]=0;
  for (i = 0; i < WSIZE/2; i++)
  {
    act_y [i] = act_z [i] = act_t [i] = 0;  // woodstock does not clear act_m, f STK will keep display format in act_m
    if(TypeFlags & TF_CLASSIC)
      act_m[i] = 0;  // clear act_m = (act_n) = 0 only classic
  }
#endif
}

void op_clear_s ()
{
  if(TypeFlags & TF_CLASSIC/* || HPType==0x01*/)
    act_s = 0; // clear all bits
  else
  	act_s&=0x8026; // clear all bits except 1,2,5,15
}

void op_display_toggle ()
{
	act_flags ^= AF_DISPLAY_ON;
}

void op_display_off ()
{
	act_flags &= ~AF_DISPLAY_ON;
  hp01flags &= ~F_BLINK;
}

void op_m_to_c()
{
#ifdef INLINEASM
#asm
  movlb   0
  bsf   _dest,6   ; force copy instruction
  ljmp  op_mx0
#endasm
#else
   	uint8_t i,j;
	for (i=j=0;i<WSIZE; i+=2,j++)
	{
		act_c[i+1] = act_m[j]>>4;
		act_c[i]   = act_m[j]&0x0f;
	}
#endif
}

void op_m_exch_c()
{
#ifdef INLINEASM
#asm
  movlb   0
  bcf   _dest,6   ; force exch instruction
op_mx0:
	movlw	low _act_m
	movwf	FSR1L
	movlw	high _act_m
	movwf	FSR1H
  ljmp  op_mx1
#endasm
#else
   	uint8_t i,j;
	for (i=j=0;i<WSIZE; i+=2,j++)
	{
		uint8_t t = (act_c[i] & 0x0f) | (act_c[i+1]<<4);
		act_c[i+1] = act_m[j]>>4;
		act_c[i]   = act_m[j]&0x0f;
		act_m[j] = t;
	}
#endif
}

void op_mx()
{
#ifdef INLINEASM
#asm
  movlb  0
	movf  _opcode,w   ; dest = opcode
  movwf _dest

	movlw	low _act_m
	btfsc	_opcode,7
	movlw	low	_act_n
	movwf	FSR1L
	movlw	high _act_m
	btfsc	_opcode,7
	movlw	high _act_n
	movwf	FSR1H
op_mx1:	
	movlw	low _act_c
	movwf	FSR0L
	movlw	high _act_c
	movwf	FSR0H
	
	movlw	7
	movwf	_first
	
l_151:

	moviw	[1]FSR0  ; act_c[i+1]<<4
	swapf	WREG,w
	iorwf	INDF0,w ; act_c[] has only values from 0-f
	movwf	_last
	
	movf	INDF1,w ; act_c[i] = pt[j]&0x0f;
	andlw	15
	movwf	INDF0	
	incf	FSR0L
	
	swapf	INDF1,w ; act_c[i+1] = pt[j]>>4;
	andlw	15
	movwf	INDF0	
	incf	FSR0L
	
	btfsc	_dest,6 ; if (!((char)opcode & (1<<6)))
	goto l_150
	
	movf	_last,w ; pt[j] = t;
	movwf	INDF1
	
l_150:
	incf	FSR1L
	decfsz	_first
	goto  l_151
#endasm
#else
	//0100, 0101, 0110, 0111
	//op_m1_exch_c, op_m1_to_c, op_m2_exch_c, op_m2_to_c,
	uint8_t i,j;
	uint8_t *pt= (char)opcode & (1<<7) ? act_n : act_m;
	for (i=j=0;i<WSIZE; i+=2,j++)
	{
		uint8_t t = (act_c[i] & 0x0f) | (act_c[i+1]<<4);
		act_c[i+1] = pt[j]>>4;
		act_c[i]   = pt[j]&0x0f;
		if (!((char)opcode & (1<<6)))
			pt[j] = t;
	}
#endif
}

void op_y_to_a ()
{
  dest = act_a; src = act_y; reg_decompress();
}

void op_stack_to_a ()
{
  op_y_to_a();
  dest=act_y; src=act_z; first=0; last=WSIZE-1; // copy 2 registers
  reg_copy();
}

void op_down_rotate ()
{
#ifdef INLINEASM
#asm
  movlb 0
	movlw	high _act_c			; all act_registers a,b,c,y,z,t are located in same bank
	movwf	FSR0H
	movwf	FSR1H
	movlw	low _act_c
	movwf	_dest
	movlw	low _act_y
	movwf	_src

l_176:
  movf  _dest,w
	movwf FSR0L
	moviw FSR0++  ; act_c[i*2]
	andlw	15
	movwf	_s2
	swapf	INDF0,w ; act_c[i*2+1]<<4
	andlw	240
	iorwf	_s2		; temp = act_c[i*2] & 0x0f | act_c[i*2+1]<<4
  incf  _dest
  incf  _dest

	movf	_src,w  ; act_y
	movwf	FSR1L
	swapf	INDF1,w
	andlw	15
	movwi	FSR0--		; act_c [i*2+1] = act_y[i]>>4;
	movf	INDF1,w
	andlw 15
	movwf	INDF0		; act_c [i*2] = act_y[i]&0x0f;

	movf	_src,w
	addlw	7
	movwf	FSR0L  ; act_z
	movf	INDF0,w
	movwf	INDF1  ; act_y [i] = act_z [i];

	movlw	14
	addwf	FSR1L	; act_t
	movf	INDF1,w
	movwf	INDF0  ; act_z [i] = act_t [i];

	movf	_s2,w
	movwf	INDF1  ; act_t [i] = temp;

	incf	_src
	movlw	low _act_z  ; 74+7
	subwf	_src,w
	skipz
	goto l_176
#endasm
#else

  uint8_t i, t;

  for (i = 0; i < WSIZE/2; i++)
    {
      t = (act_c[i*2]&0x0f) | (act_c[i*2+1]<<4);
      act_c [i*2] = act_y[i]&0x0f;
      act_c [i*2+1] = act_y[i]>>4;
      act_y [i] = act_z [i];
      act_z [i] = act_t [i];
      act_t [i] = t;
    }
#endif
}

void op_c_to_stack ()
{
// asm code funktioniert noch nicht -> debug
#ifdef INLINEASM
#asm
  movlb 0

	movlw high ACT_REGS		; all act_registers a,b,c,y,z,t are located in bank 1 at 0x2050
	movwf	FSR0H
	movwf	FSR1H
	movlw	low (ACT_REGS+49+6)	; pointer to act_z[6] src
	movwf	FSR0L
	movlw	low (ACT_REGS+56+6)	; pointer to act_t[6] dst
	movwf	FSR1L

	movlw	14        ; copy 2 registers of 7 byte
  movwf _first
l_172:
	moviw	FSR0--
	movwi	FSR1--		; act_t[i]=act_z[i]	
  decf  _first
  btfss 3,2       ; jnz
	goto l_172

	movlw	7
  movwf _first
l_173:
	swapf	INDF0,w
	andlw	240
	movwf	INDF1
	moviw	--FSR0	; act_c
	andlw	15
	iorwf	INDF1  ; act[y[i] = act_c[i*2]& 0x0f | act_c[i*2+1]<<4
  moviw	FSR0--
  moviw	FSR1--

  decf  _first
  btfss 3,2    ; jnz
	goto l_173
#endasm
#else
  uint8_t i;

  for (i = 0; i < WSIZE/2; i++)
    {
      act_t [i] = act_z [i];
      act_z [i] = act_y [i];
      act_y [i] = (act_c[i*2]&0x0f) | (act_c[i*2+1]<<4);
    }
#endif
}

// a[0] = f
void op_f_to_a ()
{
	act_a [0] = act_f;
}

// f <-> a[0]
void op_f_exch_a ()
{
// inline code has exactly same size than C code, but use it anyway
#ifdef INLINEASM
#asm
  movlb  1
  movf   BANKMASK(_act_a),w 	      ; uint8_t t = act_a [0];
  movlb   0
  movwf  _dest
  movf   _act_f,w
  movlb 1
  movwf   BANKMASK(_act_a)         ; act_a [0] = act_f;
  movlb   0
  movf   _dest,w
  movwf  _act_f         ;	act_f = t;
#endasm
#else
	uint8_t t;
	t = act_a [0];
	act_a [0] = act_f;
	act_f = t;
#endif
}


void op_circulate_a_left ()
{
#ifdef INLINEASM
#asm
  movlb  0
	movlw	low _act_a
	movwf	FSR0L
	movlw	high _act_a
	movwf	FSR0H
	movlw	13
	movwf	_first
	moviw	[13]FSR0
	movwf	_last
l_126:
	moviw	[12]FSR0
	movwi	[13]FSR0
	decf	FSR0L

	decfsz	_first
	goto	l_126
	movf	_last,w
	movwi	[13]FSR0
#endasm
#else
	uint8_t i, t;
	t = act_a [WSIZE - 1];
	for (i = WSIZE - 1; i >= 1; i--)
		act_a [i] = act_a [i - 1];
	act_a [0] = t;
#endif
}

void op_dec_p ()
{
  uint8_t size=WSIZE-1;

#ifdef HP01
  if(HPType==0x01)
    size=WSIZE01-1;
#endif
  if(TypeFlags & TF_CLASSIC)
    size=15;
  
  if(act_p)
	  act_p--;
 	else
  	act_p = size;
}

void op_inc_p ()
{
  uint8_t size=WSIZE;

#ifdef HP01
  if(HPType==0x01)
    size=WSIZE01;
#endif
  if(TypeFlags & TF_CLASSIC)
    size=16;

	if (++act_p >= size) // 14 bit register ?
	{
		act_p = 0;
		act_flags|=AF_PCARRY;
	}
}

void op_gokeys ()
{
	act_pc &= 0xFF00;
	handle_del_rom ();
	act_pc |= act_key_buf;
  CheckAddress();
}

void op_keys_to_a()
{
  uint8_t b = act_key_buf;
	act_a [2] = b >> 4;
	act_a [1] = b & 0x0f;
  CheckAddress(); // Die Abfrage spezieller Adressen kostet hier kaum Rechenzeit im Gegensatz zur Abfrage in Getopcode
}

void op_a_to_rom_addr ()
{
	act_pc &= 0xFF00;
	handle_del_rom ();
	act_pc += ((act_a [2] << 4) + act_a [1]);

  CheckAddress(); // Die Abfrage spezieller Adressen kostet hier kaum Rechenzeit im Gegensatz zur Abfrage in Getopcode
}

void op_test_s_eq_0 ()
{
	act_inst_state = branch;
	if (act_s & (1 << op6))
		act_flags |= AF_CARRY;
}

void op_test_s_eq_1 ()
{
  op_test_s_eq_0();
  act_flags^=AF_CARRY;

//	act_inst_state = branch;
//	if (!(act_s & (1 << op6)))
//		act_flags |= AF_CARRY;
}

void op_load_constant ()
{
#ifdef HP01
  if(HPType==0x01)
  {
    if (act_p < WSIZE01) // A(P) = n
      act_a[act_p] = op6;
  }
  else
#endif
  if(TypeFlags & TF_CLASSIC)
  {
    if (act_p < WSIZE && op6 < 10) // /* HP-45 depends on load constant with p > 13 not affecting C */
       act_c[act_p] = op6;
  }
  else
   	act_c [act_p] = op6;

	op_dec_p();
}

void op_sel_rom ()
{
  if(TypeFlags & TF_CLASSIC)
  {
    act_del_rom = act_del_rom & 0x08 | opcode >> 7; // set bit 8-10 keep bit 11
    act_pc = (uint16_t)(act_del_rom<<8 | (uint8_t)act_pc);
  }
  else
	  act_pc = opcode<<2 & 0xFF00 | (uint8_t)act_pc; // act_pc(high)=opcode<<2 act_pc=act_pc & 0xFF | (opcode<<2 & 0xFF00)
}

// copy act_c to RAM address specified by opcode
// external RAM interprets this instruction, sets its RAM addr according to opcode
// and reads DATA line in next cycle
void op_c_to_register ()
{
	act_ram_addr &= 0xF0;
	act_ram_addr += op6;
	op_c_to_data();
}

void op_c_to_data ()
{
	c_to_register(act_ram_addr);
}

void op_c_to_addr ()
{
  if(TypeFlags & TF_CLASSIC)
  {
    if(act_ram_size<=10)
      act_ram_addr = (uint8_t)(act_c[12]); // HP-45, HP-80
    else
      act_ram_addr = (uint8_t)(act_c[12] * 10 + act_c[11]);  // HP-55
  }
  else
  	act_ram_addr = (act_c [1] << 4) + act_c [0];
}

void op_test_p_eq ()
{
	act_inst_state = branch;
	act_flags |= AF_CARRY;
	if (act_p == p_test_map [op6])
		act_flags &= ~AF_CARRY;
}

void op_test_p_ne ()
{
#ifdef HP01
  if(HPType==0x01)
  {
    if(act_p == p_test_map01[op6]) // jump if not carry
      act_flags |= AF_CARRY;
  }
#else
  {
	act_inst_state = branch;
	uint8_t n= p_test_map [op6];

// include special case label search  if p # 0 followed directly after p + 1 -> p
	if(act_p == n || (act_flags & AF_PCARRY && (act_p==1) && n==0))
		act_flags |= AF_CARRY;
  }
#endif
}

// 16 Register lschen ab ram_addr
void op_clear_data_regs ()
{
// it is assumed, that this opcode is ignored by external RAM in HP-25C
	uint8_t i;
	uint8_t baseaddr = act_ram_addr & 0xF0; // must be multiple of 16 registers 0x10, 0x20 etc 
	if(/*baseaddr>=0 &&*/ baseaddr<RAMBANKS*RAMBANKSIZE)
	{
		for (i = 0; i < 2; i++) // clear two banks
		{
	    getrampt(baseaddr+i*8);
	    first=0; last=RAMBANKSIZE*sizeof(creg_t)-1;
  	  reg_zero();
  	}  
	}
}

void op_del_sel_rom ()
{
// Classic set3 3 bit, others 4 bit
 if(TypeFlags & TF_CLASSIC)
    act_del_rom = (act_del_rom & 0x08) | (opcode >> 7);  // set bit 8-10, keep bit 11
 else
 {
	  act_del_rom = op6;  // set 4-bit upper pc delayed
    act_flags |= AF_DEL_ROM;
 }
}

void op_register_to_c ()
{
	if(op6)
		act_ram_addr = (act_ram_addr & 0xF0) | op6;
	register_to_c(act_ram_addr);
}

// p = immediate value 0-15
void op_set_p ()
{
  if(TypeFlags & TF_CLASSIC)
   act_p = op6;
  else
	  act_p = p_set_map [op6];
}

// now follow opcodes not part of woodstock ACT

// only CLASSIC command
void op_data_to_c()
{
  register_to_c(act_ram_addr);
}

void op_del_sel_grp()
{
  act_del_rom=(act_del_rom & 0x07) | (opcode >> 4 & 0x08); // set bit 11 keep bit 8-10
}

void op_test_p()
{
  if(act_p == op6)
    act_flags |= AF_CARRY;
}

// following instructions are only used by HP-01 firmare
#ifdef HP01
void op_display_on ()
{
	act_flags |= AF_DISPLAY_ON;
}

void op_clr_s17()
{
  act_s &=0xFF00;
}

void op_clr_s815()
{
  act_s &=0x00FF;
}

void op_sleep()
{
  hp01flags |= F_SLEEP;
}

void op_blink()
{
  hp01flags |= F_BLINK;
}

void op_enscwp()
{
  hp01flags |= F_SCWP;
}

void op_dsscwp()
{
  hp01flags &= ~F_SCWP;
}

void op_swinc()
{
  hp01flags &= ~F_SWDEC;
}

void op_swdec()
{
  hp01flags |= F_SWDEC;
}

void op_swstrt()
{
  hp01flags |= F_SWSTARTED;
}

void op_swstop()
{
  hp01flags &= ~F_SWSTARTED;
}

void op_altog()
{
  hp01flags ^= F_ALARMACTIV;
}

void op_cdex()
{
  first = 0; last = WSIZE/2 - 1;
  dest = act_reg; src = act_y; reg_copy();  // make a copy of act_y
  dest = act_y; src = act_c; reg_compress(); // copy act_c to act_y
  dest = act_c; src = act_reg; reg_decompress(); // copy temporary reg to act_c
}

void op_mtoc()
{
  dest = act_c; src = act_m; reg_decompress();
}

void op_ctom()
{
  dest = act_m; src = act_c; reg_compress();
}

void op_dtoc()
{
  dest = act_c; src = act_y; reg_decompress();
}

void op_ftoap()
{
  act_a[act_p]=act_f;
}

void op_aptof()
{
  act_f=act_a[act_p];
}

void op_dsptoa()
{
  first = 0; last = WSIZE01 - 1;
  dest = act_a; src = act_dsp; reg_copy();
}

void op_cltoa()
{
  first = 0; last = WSIZE01 - 1;
  dest = act_a; src = act_cl; reg_copy();
}

void op_atocl()
{
  first = 0; last = WSIZE01 - 1;
  dest = act_cl; src = act_a; reg_copy();
}

// calculator has no accurate time, reset fraction of seconds not available
void op_atoclrs()
{
	op_atocl();
}

// cl register has different format than dsp,  6 bytes HHMMSS always 24 hours
// 12/24 hours formatting is made by WAKEUP every hour
// supress leading zeros
void op_cltodsp()
{
  act_dsp[4] = act_cl[0];
  act_dsp[5] = act_cl[1];
  act_dsp[7] = act_cl[2];
  act_dsp[8] = act_cl[3];
}

static void op_atoal()
{
  first = 0; last = WSIZE01 - 1;
  dest = act_al; src = act_a; reg_copy();
  hp01flags ^= F_ALARMACTIV;
}

static void op_swtoa()
{
  first = 0; last = WSIZE01 - 1;
  dest = act_a; src = act_sw; reg_copy();
}

static void op_atosw()
{
  first = 0; last = WSIZE01 - 1;
  dest = act_sw; src = act_a; reg_copy();
}

static void op_atodsp()
{
  first = 0; last = WSIZE01 - 1;
  dest = act_dsp; src = act_a; reg_copy();
}

void op_altoa()
{
  first = 0; last = WSIZE01 - 1;
  dest = act_a; src = act_al; reg_copy();
}

// Stopwatch register is 6 digits MM:SS.CC
void op_swtodsp()
{
  uint8_t i,j,k;
  k= (act_sw[6] | act_sw[7])==0 ? 0 : 2;// hours 0? show 1/100 else hours
  for(j=3,i=0;i<6;i++,j++)
  {
     act_dsp[j]=act_sw[i+k]; // optional use all 12 digits in HP-25 HH:MM:SS, original only HH:MM:SS can be shown
     if(i & 1) j++;  // skip display digit every two digits
  }
}

static void op_altodsp()
{
  if(hp01flags & F_ALARMACTIV)
    if(act_dsp[3]==0x0A)
     act_dsp[3]=0x0E;  // Alarm Dot and Minus
    else
     act_dsp[3]=0x0B;  // Alarm Minus

  act_dsp[4] = act_al[0]; // copy only MM:SS
  act_dsp[5] = act_al[1]; // HH will be set by software
  act_dsp[7] = act_al[2];
  act_dsp[8] = act_al[3]; 
}
#endif // HP01

//#endif // FLASHFCNTABLE

// Hilfsfunktionen, die in assembler vorhanden sind
// register lschen 
void reg_zero ()
{
#ifdef INLINEASM
#asm
	lcall	loadFSR1   ; FSR1 = dest+first
l_100:
	movf	_first,w
	subwf	_last,w
	skipc          ; return if first>last
	return

	movlw	0        ; *dest++=0
  movwi FSR1++
	incf	_first,f
	goto	l_100

loadFSR0:
  movlb  0
	movf	_src,w
	addwf	_first,w ; cannot overflow, because no act registers and first 8 RAM registers crosses boundary 
	movwf	FSR0L
	movf	_src+1,w
	movwf	FSR0H

; load 16-bit pointer FSR1 = _dest+first
loadFSR1:
  movlb  0
	movf	_dest,w
	addwf	_first,w  ; cannot overflow, because no act registers  and first 8 RAM registers crosses boundary 
	movwf	FSR1L
	movf	_dest+1,w
	movwf	FSR1H
//	return
#endasm
#else
	uint8_t i;
	for (i = first; i <= last; i++) dest [i] = 0;
#endif
}

void reg_copy()
{
#ifdef INLINEASM
#asm
	lcall	loadFSR0   ; FSR0 = src+first,  FSR1 = dest+first

l_102:
	movf	_first,w
	subwf	_last,w
	btfss	3,0
	return

	moviw	FSR0++    ; *dest++=*scr++
	movwi	FSR1++
	incf	_first,f
	goto	l_102
#endasm
#else
	uint8_t i;
	for (i = first; i <= last; i++) dest [i] = src [i];
#endif
}


void reg_exch ()
{
#ifdef INLINEASM
#asm
	lcall	loadFSR0

l_104:
	movf	_first,w
	subwf	_last,w
	btfss	3,0
	return

	movf	INDF0,w
	movwf	_dest  ; use dest as temp variable
	movf	INDF1,w
	movwf	INDF0
	movf	_dest,w
	movwf	INDF1

	incf    FSR0L,f
	incf    FSR1L,f
	incf	_first,f
	goto	l_104
#endasm
#else
	uint8_t i, t;
	for (i = first; i <= last; i++)
	{
		t = dest [i];
		dest [i] = src [i];
		src [i] = t;
	}
#endif
}

void reg_shift_right()
{
#ifdef INLINEASM
#asm
	lcall	loadFSR1

l_106:
	movlw	0
	movwi	[0]FSR1  ; last byte set to 0

	movf	_first,w
	subwf	_last,w
	btfsc	3,2      ; return if first==last
	return

	moviw	[1]FSR1
	movwi	[0]FSR1

	incf    FSR1L,f
	incf	_first,f
	goto	l_106
#endasm
#else
	uint8_t i;
	for (i = first; i <= last; i++) dest [i] = (i == last) ? 0 : dest [i+1];
#endif
}

void reg_shift_left()
{
#ifdef INLINEASM
#asm
  movlb  0
	movf	_dest,w
	addwf	_last,w
	movwf	FSR1L
	movf	_dest+1,w
	movwf	FSR1H

l_108:
	movlw	0
	movwi	[0]FSR1  ; last byte set to 0

	movf	_first,w
	subwf	_last,w
	btfsc	3,2      ; return if first==last
	return

	moviw	[-1]FSR1
	movwi	[0]FSR1

	decf    FSR1L,f
	decf	_last,f
	goto	l_108
#endasm
#else
	int8_t i;
	for (i = last; i >= first; i--) dest [i] = (i == first) ? 0 : dest [i-1];
#endif
}

void reg_inc()
{
	act_flags |= AF_CARRY;
	src=NULL;
	reg_add ();
}

void reg_add()
{
#ifdef INLINEASM
#asm
	lcall	loadFSR0

l_110:
	movf	_first,w
	subwf	_last,w
	btfss	3,0
	return

	movf	_src,w  ; if (src==NULL)
	iorwf	_src+1,w
	skipz
	movf	INDF0,w
	addwf INDF1,f  ; dest = dest + src
	btfsc	_act_flags,1 ; act carry set ?
	incf	INDF1

	bcf		_act_flags,1
	movf	_base,w
	subwf	INDF1,w
	skipc
	goto l_111

	movwf	INDF1    ; dest = dest - base
	bsf		_act_flags,1
	
l_111:

	incf	FSR0L,f
	incf  FSR1L,f
	incf	_first,f
	goto	l_110
#endasm
#else
	uint8_t i,res;
	for (i = first; i <= last; i++)
	{
		uint8_t s = src ? src[i] : 0;
		res = dest[i] + s + (act_flags & AF_CARRY ? 1 : 0);
		if (res >= base)
		{
			res -= base;
			act_flags |= AF_CARRY;
		}
		else
		{
			act_flags &= ~AF_CARRY;
		}
		dest[i]=res;
	}
#endif
}

void reg_sub ()
{
#ifdef INLINEASM
#asm
  movlb  0
	movf	_src,w                          
	addwf	_first,w
	movwf	FSR0L
	movf	_src+1,w
	movwf	FSR0H
	iorwf	_src

l_114:

	movf	_src2,w
	addwf	_first,w
	movwf	FSR1L
	movf	_src2+1,w
	movwf	FSR1H
	iorwf	_src2,w
	movwf	_s2

	movf	_first,w
	subwf	_last,w
	btfss	3,0
	return

	movlw	0
	movf	_src,f     ; if(src==NULL)
	skipz
	movwf  	INDF0,w  ; w = src - src2 - carry
	movwf	_src+1
	
	movlw	0
	movf	_s2,f     ; if(src2==NULL)
	skipz
	movf	INDF1,w
	subwf	_src+1,w   ; -s2

	
	btfsc	_act_flags,1 ; act carry set ?
	decf	WREG,w
	bcf		_act_flags,1
	btfss	WREG,7  ; d <0 ?
	goto	l_115
	
	addwf	_base,w  ; w = w + base
	bsf		_act_flags,1

l_115:
	movwf	_src+1   ; src+1 = w

	movf	_dest,w
	iorwf	_dest+1,w
	skipnz
	goto l_117

	movf	_dest,w
	addwf	_first,w
	movwf	FSR1L
	movf	_dest+1,w
	movwf	FSR1H
	movf	_src+1,w
	movwf	INDF1   ; if(dest) dest = w

l_117:
	incf	FSR0L,f
	incf	_first,f
	goto	l_114
#endasm
#else

	uint8_t i;
	for (i = first; i <= last; i++)
	{
		uint8_t s1 = src ? src[i] : 0;
		uint8_t s2 = src2 ? src2[i] : 0;

		signed char d = (s1 - s2) - (act_flags & AF_CARRY ? 1 : 0);
		if (d < 0)
		{
			d += base;
			act_flags |= AF_CARRY;
		}
		else {
			act_flags &= ~AF_CARRY;
		}
		if (dest) dest [i] = d;
	}
#endif
}

void reg_test_equal ()
{
#ifdef INLINEASM
#asm
	lcall	loadFSR0

	bsf		_act_flags,1
l_116:
	movf	_first,w
	subwf	_last,w
	btfss	3,0
	return
	
	movf	INDF0,w
	incf	FSR0L
	incf	_first
	xorlw	0
	btfsc	3,2
	goto	l_116
	bcf		_act_flags,1
#endasm
#else
	uint8_t i,d;

	act_flags |= AF_CARRY;
	for (i = first; i <= last; i++)
	{
        d = (uint8_t)(dest == NULL ? 0 : dest[i]);
		if (/*(act_flags & AF_CARRY) &&*/ (src [i] != d))
		{
			act_flags &= ~AF_CARRY;
			break;
		}
	}
#endif
}

void reg_test_nonequal ()
{
#ifdef INLINEASM
#asm
	lcall	loadFSR0

l_118:
	movf	_first,w
	subwf	_last,w
	btfss	3,0
	return
	
	movf	INDF0,w
	incf	FSR0L
	incf	_first
	xorlw	0
	btfsc	3,2
	goto	l_118
	bsf		_act_flags,1
#endasm
#else
	uint8_t i,d;

	act_flags &= ~AF_CARRY;
	for (i = first; i <= last; i++)
	{
        d = (uint8_t)(dest == NULL ? 0 : dest[i]);
		if (src [i] != d)
		{
			act_flags |= AF_CARRY;
			break;
		}
	}
#endif
}

void setfield(uint8_t field)
{
#ifdef INLINEASM
#asm
  movlb  0
  movwf  _first
  movlw  13    ; most used WSIZE-1
  movwf _last
  movf  _first,f
	skipz
	goto setfield1

setfield0:				; if(field==0)
	movf	_act_p,w	; /* p  */ first = act_p; last = act_p;
	movwf	_first
  goto setlast

setfield1:				
	decfsz	_first
	goto	setfield2	; if(field==1)
	movf	_act_p,w	; /* wp */ first = 0; last = act_p;
  goto setlast

setfield2:
	decfsz	_first
	goto	setfield3	; if(field==2)
	movlw	2			;  /* xs */ first = EXPSIZE - 1; last = EXPSIZE - 1; 
	movwf	_first
  goto  setlast

; QC passed 29.10.2017
setfield3:
	decfsz	_first
	goto	setfield4	; if(field==3)
	movlw	2			; /* x  */  first = 0; last = EXPSIZE - 1;
setlast:
	movwf	_last
	return

setfield4:
	decfsz	_first
	goto	setfield5	; if(field==4)
	movlw	13			; /* s  */  first = WSIZE - 1;   last = WSIZE - 1;
	movwf	_first
	return

setfield5:
	decfsz	_first
	goto	setfield6	; if(field==5)
	movlw	3			; /* m  */  first = EXPSIZE; last = WSIZE - 2;
	movwf	_first
	decf	_last
	return

setfield6:
	decfsz	_first
	goto	setfield7	; if(field==6)
	return		; /* w  */  first = 0; last = WSIZE - 1;

setfield7:				; if(field==7)
	movlw	3			;  /* ms */  first = EXPSIZE; last = WSIZE - 1; 
	movwf	_first
	return
#endasm
#else

   	switch (field)
   	{
		case 0:  /* p  */  first = act_p; last = act_p; /* if (act_p >= WSIZE) last = 0; */ break;
		case 1:  /* wp */  first = 0; last = act_p; /*if (act_p >= WSIZE) last = WSIZE - 1; */ break;
		case 2:  /* xs */  first = EXPSIZE - 1; last = EXPSIZE - 1; break;
		case 3:  /* x  */  first = 0;           last = EXPSIZE - 1; break;
		case 4:  /* s  */  first = WSIZE - 1;   last = WSIZE - 1;   break;
		case 5:  /* m  */  first = EXPSIZE;     last = WSIZE - 2;   break;
		case 6:  /* w  */  first = 0;           last = WSIZE - 1;   break;
		case 7:  /* ms */  first = EXPSIZE;     last = WSIZE - 1;   break;
    }
#endif
}

// following op_del_sel_rom instruction upper 4 bits of pc will be set after goto jsb a_to_romaddr gokeys
// act_del_rom =4 bit and 3-bit for classic
// HP-35/45/80 doesnt use del_sel_rom HP-55/70 does use,
// hp-65 uses del_sel_rom 0-2 and del_sel_grp 0-1
// hp-55 uses del_sel_rom 0-7 and del_sel_grp 0-1
// HP-21/22/25/27/29 and spice use 
// HP-25 uses delayed rom 0-7, HP-29 0-15

void handle_del_rom()
{
	if (act_flags & AF_DEL_ROM)
	{
		act_pc = (act_del_rom << 8) | (uint8_t)act_pc; // generate 12-bit pc
		act_flags &= ~AF_DEL_ROM;
	}
}

// jump to subroutine
void op_jsb()
{
  if(TypeFlags & TF_CLASSIC)
  {
    act_stack[0] = act_pc; // classic has stack depth 1
    act_flags|=AF_DEL_ROM;
  }
  else
  {
    act_stack[act_sp] = act_pc;
	  act_sp = (act_sp+1) & (STACK_SIZE-1);
  }
	act_pc = (act_pc & 0xFF00) | opcode >> 2;
	handle_del_rom ();
}

void op_return()
{
  if(TypeFlags & TF_CLASSIC)
    act_pcl= (uint8_t)act_stack[0]; // leave act_rom and act_grp bits
  else
  {
	  act_sp = (act_sp-1) & (STACK_SIZE - 1);
	  act_pc = act_stack [act_sp];
  }
  CheckAddress(); // Die Abfrage spezieller Adressen kostet hier kaum Rechenzeit im Gegensatz zur Abfrage in Getopcode
}


// goto absolute 8-bit address
void op_goto()
{
  if ((act_flags & AF_PREV_CARRY) == 0)
  {

    act_pc = (act_pc & 0xFF00) | (opcode >> 2);
    if(TypeFlags & TF_CLASSIC)
      act_flags|=AF_DEL_ROM;
    handle_del_rom();
  }
}

void reg_compress()
{
#ifdef INLINEASM
#asm
  movlb  0
	clrf	_first
	lcall	loadFSR0

	movlw	7  ; for (uint8_t i=0;i<WSIZE/2;i++)
	movwf	_dest

l_120:
	moviw	FSR0++ ; dest[i] = src[i*2] & 0x0f | src[i*2+1]<<4 ;
	andlw	15
	movwf	_src      ; temporary reg
	swapf	INDF0,w
	andlw	240
	iorwf	_src,w
  movwi FSR1++
  moviw FSR0++  // just increment w not used
	decfsz	_dest
	goto 	l_120
	return
#endasm
#else
	for (uint8_t i=0;i<WSIZE/2;i++)
		dest[i] = (src[i*2+1]<<4) | (src[i*2] & 0x0f);
#endif
}

void reg_decompress()
{
#ifdef INLINEASM
#asm
  movlb  0
	clrf	_first
	lcall	loadFSR0

	movlw	7  ; for (uint8_t i=0;i<WSIZE/2;i++)
	movwf	_dest

l_122:
	moviw	[0]FSR0    ; dest[i*2]   = src[i] & 0x0F;
	andlw	15
	movwi	FSR1++  ; dest[i*2+1] = src[i] >>4;
	swapf	INDF0,w
	andlw	15
	movwi	FSR1++
	moviw	FSR0++  ; increment w not used
	
	decfsz	_dest
	goto 	l_122
	return

#endasm
#else
	for (uint8_t i=0;i<WSIZE/2;i++)
	{
		dest[i*2]   = src[i] & 0x0F;
		dest[i*2+1] = src[i] >>4;
	}
#endif
}

// recall register n(0-63) from rambank to uncompressed destination register act_c 
// external RAM interprets this instruction and outputs RAM contents to DATA line the next cycle
void register_to_c(uint8_t addr)
{
	if(addr<0x40) // limit to 64 registers
	{
		getrampt(addr);	src=dest;
		dest=act_c; 
		reg_decompress();
	}
}

void c_to_register(uint8_t addr)
{
	if(addr<0x40) // limit to 64 registers
	{
		src=act_c;
		getrampt(addr); // dest = act_ram[addr]
		reg_compress();
	}
}

void m_to_register(uint8_t addr)
{
		src=act_m;
    getrampt(addr); // dest=act_ram[addr]
    first=0; last = WSIZE/2-1;
		reg_copy();
}

void register_to_m(uint8_t addr)
{
    getrampt(addr);  // dest=act_ram[addr]
    src=dest;
		dest=act_m;
    first=0; last = WSIZE/2-1;
		reg_copy();
}

#ifndef USEASM
// getrampt(UCHAR addr) calculate pointer to act_ram[addr] into dest
// each element of creg_t act_ram[] is 7 byte
// addr : Index to act_ram[] 0-63
void getrampt(uint8_t addr)
{
#ifdef INLINEASM
#asm
	movlb  0
	movwf  _dest   ; dest=addr
  clrf   _dest+1,f
	lslf	 _dest,f   ; dest = addr*8
	rlf	   _dest+1,f
	lslf	 _dest,f
	rlf	   _dest+1,f
	lslf	 _dest,f
	rlf	   _dest+1,f
	subwf  _dest,f   ; dest-=addr  -> dest=addr*7
  skipc
  decf   _dest+1,f

  movlw	  160     ; dest+=0x20A0
  addwf	 _dest,f
  movlw   32
  addwfc _dest+1,f
#endasm
#else
  dest=(uint8_t *)(addr*8 - addr + ACT_RAM);
#endif
}
#endif

#endif // USEASM

// normalize number
// 14 digits (each digit 4-bit) number format from left to right
// sign 0=positive 9=negative
// 10 digits mantissa no leading zero
// sign exponent 0=positive 9=negative
// 2 digits exponent  99=1e99  999=1e-1
//  6 5 4 3 2 1 0
// -1234567890-12
// examples
// 0 1000000000 000  = 1.000000
// 0 1234567000 000  = 1.234567
// 0 2345670000 001  = 23.4567
// 0 2345670000 999  = 2.34567 e-01   
// 9 2345670000 002  = -234.567
void reg_normalize(uint8_t *reg)
{
uint8_t i,b,fl;

  fl=act_flags; // preserve flags if normalize is called outside of emulator
	b=base;
	base=10; // for reg_sub
	for(i=WSIZE-2;i>=EXPSIZE;i--) // loop over mantissa
	{
		if(reg[i]!=0 || i==EXPSIZE) // find first non zero, no action if mantissa is all zeros
		{
			for(;i<WSIZE-2;i++)  // leading zeros must be normalized
			{
				dest=reg;	first=EXPSIZE; last=WSIZE-2;  // keep sign
				reg_shift_left();     // shift left mantissa
				dest=reg; src=reg; src2=NULL; first=0; last=EXPSIZE-1;
				act_flags |= AF_CARRY; // decrement exponent
				reg_sub ();
			}
			break;
		}
	}
	base=b;
  act_flags=fl;
}

// die 32 arithmetik instructions sind bei Classic und HP-01 identisch aber anders angeordnet als bei Woodstock ACT
const uint8_t ArithmetikTableClassic[] =
{
0x16,0x01,0x18,0x1b,0x06,0x14,0x08,0x15,
0x0e,0x03,0x11,0x13,0x05,0x17,0x0c,0x0f,
0x19,0x07,0x1f,0x1a,0x1e,0x0b,0x1d,0x00,
0x10,0x02,0x1c,0x12,0x09,0x04,0x0a,0x0d,
};

#ifdef HP01
const uint8_t ArithmetikTable01[] =
{
0x17,0x08,0x0f,0x13,0x1b,0x0b,0x14,0x15,
0x04,0x05,0x0a,0x1c,0x18,0x1a,0x0c,0x11,
0x19,0x00,0x0d,0x12,0x03,0x02,0x09,0x10,
0x01,0x16,0x06,0x07,0x1d,0x0e,0x1e,0x1f,
};
#endif

const uint8_t FieldIndexTable[8] = { 0,5,3,6,1,7,2,4 };

void op_arith()
{
  uint8_t field = (opcode >> 2) & 7;

  if(TypeFlags & TF_CLASSIC /* || HPType==0x01*/)
    field  = FieldIndexTable[field];  // classic calculators have different indexes

  setfield(field);

	act_flags &= ~AF_CARRY;  // carry was set to 0 by execute_instruction before

  uint8_t op = opcode >> 5;

#ifdef HP01
  if(HPType==0x01)
    op=ArithmetikTable01[op];
#endif
  if(TypeFlags & TF_CLASSIC)
    op=ArithmetikTableClassic[op];

  switch (op) // HP-21-29/31-38 ACT arithmetic instructions
  {
		case 0x00:  /* 0 -> a[f] */
			dest=act_a; reg_zero (); break;
		case 0x01:  /* 0 -> b[f] */
			dest=act_b;	reg_zero (); break;
		case 0x02:  /* a exchange b[f] */
			src=act_b; dest=act_a;	reg_exch(); break;
		case 0x03:  /* a -> b[f] */
			src=act_a; dest=act_b;	reg_copy(); break;
		case 0x04:  /* a exchange c[f] */
			src=act_c; dest=act_a;	reg_exch(); break;
		case 0x05:  /* c -> a[f] */
			src=act_c; dest=act_a;	reg_copy(); break;
		case 0x06:  /* b -> c[f] */
			src=act_b; dest=act_c;	reg_copy(); break;
		case 0x07:  /* b exchange c[f] */
			src=act_c; dest=act_b;	reg_exch(); break;
		case 0x08:  /* 0 -> c[f] */
			dest=act_c;	reg_zero (); break;
		case 0x09:  /* a + b -> a[f] */
			dest=act_a; src=act_b; reg_add(); break;
		case 0x0a:  /* a + c -> a[f] */
			dest=act_a; src=act_c; reg_add(); break;
		case 0x0b:  /* c + c -> c[f] */
			dest=act_c; src=act_c; reg_add(); break;
		case 0x0c:  /* a + c -> c[f] */
			dest=act_c; src=act_a; reg_add(); break;
		case 0x0d:  /* a + 1 -> a[f] */
			dest=act_a; reg_inc(); break;
		case 0x0e:  /* shift left a[f] */
			dest=act_a;	reg_shift_left(); break;
		case 0x0f:  /* c + 1 -> c[f] */
			dest=act_c; reg_inc(); break;
		case 0x10:  /* a - b -> a[f] */
			dest=act_a; src=act_a;	src2=act_b; reg_sub (); break;
		case 0x11:  /* a - c -> c[f] */
			dest=act_c; src=act_a;	src2= act_c; reg_sub (); break;
		case 0x12:  /* a - 1 -> a[f] */
			act_flags |= AF_CARRY; dest=act_a; src=act_a; src2=NULL; reg_sub (); break;
		case 0x13:  /* c - 1 -> c[f] */
			act_flags |= AF_CARRY; dest=act_c; src=act_c; src2=NULL; reg_sub (); break;
		case 0x14:  /* 0 - c -> c[f] */
			dest=act_c;	src=NULL; src2=act_c; reg_sub (); break;
		case 0x15:  /* 0 - c - 1 -> c[f] */
			act_flags |= AF_CARRY; dest=act_c; src=NULL; src2=act_c; reg_sub (); break;
		case 0x16:  /* if b[f] = 0 */
			act_inst_state = branch; src=act_b;	dest=NULL; reg_test_nonequal (); break;
		case 0x17:  /* if c[f] = 0 */
			act_inst_state = branch; src=act_c;	dest=NULL; reg_test_nonequal (); break;
		case 0x18:  /* if a >= c[f] */
			act_inst_state = branch; dest=NULL; src=act_a; src2=act_c; reg_sub (); break;
		case 0x19:  /* if a >= b[f] */
			act_inst_state = branch; dest=NULL; src=act_a; src2=act_b; reg_sub ();	break;
		case 0x1a:  /* if a[f] # 0 */
			act_inst_state = branch; src=act_a;	dest=NULL; reg_test_equal (); break;
		case 0x1b:  /* if c[f] # 0 */
			act_inst_state = branch; src=act_c;	dest=NULL; reg_test_equal ();	break;
		case 0x1c:  /* a - c -> a[f] */
			dest=act_a; src=act_a; src2=act_c; reg_sub (); break;
		case 0x1d:  /* shift right a[f] */
			dest=act_a;	reg_shift_right(); break;
		case 0x1e:  /* shift right b[f] */
			dest=act_b;	reg_shift_right(); break;
		case 0x1f:  /* shift right c[f] */
			dest=act_c;	reg_shift_right(); break;
  }
}

// only SPICE command
void op_rom_selftest()
{
//	act_inst_state = selftest;
//	act_pc &= 0xFC00;  // start from beginning of current 1K ROM bank
}

void rom_selftest_done()
{
	// ROM self-test completed, return and set S5 if error
	//if (act_crc != 0x078) act_s [5] = 1;  // indicate error
	act_inst_state = norm;
	op_return ();
}

#ifndef FLASHFCNTABLE

#ifdef HP01
static void (* const _op_fcn_0000 [256])() =
{
	op_nop    , op_set_s, op_nop, op_set_p, op_clr_s17 , op_test_s_eq_0, op_load_constant, op_clear_reg, op_sel_rom, op_clr_s, op_unknown, op_test_p_ne, op_unknown, op_del_sel_rom,op_unknown, op_cltoa,
	op_unknown, op_set_s, op_nop, op_set_p, op_clr_s815, op_test_s_eq_0, op_load_constant, op_cdex     , op_sel_rom, op_clr_s, op_unknown, op_test_p_ne, op_unknown, op_del_sel_rom,op_unknown, op_atoclrs, 
	op_unknown, op_set_s, op_nop, op_set_p, op_gokeys  , op_test_s_eq_0, op_load_constant, op_mtoc     , op_sel_rom, op_clr_s, op_unknown, op_test_p_ne, op_unknown, op_del_sel_rom,op_unknown, op_atocl, 
	op_unknown, op_set_s, op_nop, op_set_p, op_inc_p   , op_test_s_eq_0, op_load_constant, op_dtoc     , op_sel_rom, op_clr_s, op_unknown, op_test_p_ne, op_unknown, op_del_sel_rom,op_unknown, op_cltodsp, 
	op_unknown, op_set_s, op_nop, op_set_p, op_dec_p   , op_test_s_eq_0, op_load_constant, op_ctom     , op_sel_rom, op_clr_s, op_unknown, op_test_p_ne, op_unknown, op_del_sel_rom,op_unknown, op_atoal, 
	op_unknown, op_set_s, op_nop, op_set_p, op_return  , op_test_s_eq_0, op_load_constant, op_unknown  , op_sel_rom, op_clr_s, op_unknown, op_test_p_ne, op_unknown, op_del_sel_rom,op_unknown, op_swtoa, 
	op_unknown, op_set_s, op_nop, op_set_p, op_sleep   , op_test_s_eq_0, op_load_constant, op_unknown  , op_sel_rom, op_clr_s, op_unknown, op_test_p_ne, op_unknown, op_del_sel_rom,op_unknown, op_atosw, 
	op_unknown, op_set_s, op_nop, op_set_p, op_nop     , op_test_s_eq_0, op_load_constant, op_display_on,op_sel_rom, op_clr_s, op_unknown, op_test_p_ne, op_unknown, op_del_sel_rom,op_unknown, op_swtodsp, 
	op_unknown, op_set_s, op_nop, op_set_p, op_nop     , op_test_s_eq_0, op_load_constant, op_display_off,op_unknown,op_clr_s, op_unknown, op_test_p_ne, op_unknown, op_del_sel_rom,op_unknown, op_swdec, 
	op_unknown, op_set_s, op_nop, op_set_p, op_nop     , op_test_s_eq_0, op_load_constant, op_blink    , op_unknown, op_clr_s, op_unknown, op_test_p_ne, op_unknown, op_del_sel_rom,op_unknown, op_altodsp,
	op_unknown, op_set_s, op_nop, op_set_p, op_nop     , op_test_s_eq_0, op_load_constant, op_ftoap    , op_unknown, op_clr_s, op_unknown, op_test_p_ne, op_unknown, op_del_sel_rom,op_unknown, op_swinc, 
	op_unknown, op_set_s, op_nop, op_set_p, op_nop     , op_test_s_eq_0, op_load_constant, op_aptof    , op_unknown, op_clr_s, op_unknown, op_test_p_ne, op_unknown, op_del_sel_rom,op_unknown, op_atodsp, 
	op_unknown, op_set_s, op_nop, op_set_p, op_nop     , op_test_s_eq_0, op_load_constant, op_enscwp   , op_unknown, op_clr_s, op_unknown, op_test_p_ne, op_unknown, op_del_sel_rom,op_unknown, op_altoa, 
	op_unknown, op_set_s, op_nop, op_set_p, op_nop     , op_test_s_eq_0, op_load_constant, op_dsscwp   , op_unknown, op_clr_s, op_unknown, op_test_p_ne, op_unknown, op_del_sel_rom,op_unknown, op_swstrt, 
	op_unknown, op_set_s, op_nop, op_set_p, op_nop     , op_test_s_eq_0, op_load_constant, op_unknown  , op_unknown, op_clr_s, op_unknown, op_test_p_ne, op_unknown, op_del_sel_rom,op_unknown, op_swstop, 
	op_unknown, op_set_s, op_nop, op_set_p, op_nop     , op_test_s_eq_0, op_load_constant, op_dsptoa   , op_unknown, op_clr_s, op_unknown, op_test_p_ne, op_unknown, op_del_sel_rom,op_unknown, op_altog, 
};

#else // HP01

#if 0
static void (* const _op_fcn_0000Classic [256])() =
{
	op_nop    , op_set_s, op_nop, op_set_p, op_sel_rom, op_test_s_eq_0, op_load_constant, op_dec_p,   op_unknown, op_clr_s, op_display_toggle,op_test_p, op_return,    op_clear_s,    op_unknown,  op_inc_p,
	op_unknown, op_set_s, op_nop, op_set_p, op_sel_rom, op_test_s_eq_0, op_load_constant, op_unknown, op_unknown, op_clr_s, op_unknown,       op_test_p, op_unknown,   op_del_sel_rom,op_unknown,  op_unknown, 
	op_unknown, op_set_s, op_nop, op_set_p, op_sel_rom, op_test_s_eq_0, op_load_constant, op_unknown, op_unknown, op_clr_s, op_m_exch_c,      op_test_p, op_unknown,   op_unknown,    op_unknown,  op_unknown, 
	op_unknown, op_set_s, op_nop, op_set_p, op_gokeys , op_test_s_eq_0, op_load_constant, op_unknown, op_unknown, op_clr_s, op_unknown,       op_test_p, op_unknown,   op_del_sel_rom,op_unknown,  op_unknown, 
	op_unknown, op_set_s, op_nop, op_set_p, op_sel_rom, op_test_s_eq_0, op_load_constant, op_unknown, op_unknown, op_clr_s, op_c_to_stack,    op_test_p, op_unknown,   op_unknown,    op_unknown,  op_unknown, 
	op_unknown, op_set_s, op_nop, op_set_p, op_sel_rom, op_test_s_eq_0, op_load_constant, op_unknown, op_unknown, op_clr_s, op_unknown,       op_test_p, op_unknown,   op_del_sel_rom,op_unknown,  op_unknown, 
	op_unknown, op_set_s, op_nop, op_set_p, op_sel_rom, op_test_s_eq_0, op_load_constant, op_unknown, op_unknown, op_clr_s, op_stack_to_a,    op_test_p, op_unknown,   op_unknown,    op_unknown,  op_unknown, 
	op_unknown, op_set_s, op_nop, op_set_p, op_sel_rom, op_test_s_eq_0, op_load_constant, op_unknown, op_unknown, op_clr_s, op_unknown,       op_test_p, op_unknown,   op_del_sel_rom,op_unknown,  op_unknown, 
	op_unknown, op_set_s, op_nop, op_set_p, op_sel_rom, op_test_s_eq_0, op_load_constant, op_unknown, op_unknown, op_clr_s, op_display_off,   op_test_p, op_unknown,   op_del_sel_grp,op_unknown,  op_unknown, 
	op_unknown, op_set_s, op_nop, op_set_p, op_sel_rom, op_test_s_eq_0, op_load_constant, op_unknown, op_unknown, op_clr_s, op_unknown,       op_test_p, op_c_to_addr, op_del_sel_rom,op_unknown,  op_unknown,
	op_unknown, op_set_s, op_nop, op_set_p, op_sel_rom, op_test_s_eq_0, op_unknown      , op_unknown, op_unknown, op_clr_s, op_m_to_c,        op_test_p, op_unknown,   op_del_sel_grp,op_unknown,  op_unknown, 
	op_unknown, op_set_s, op_nop, op_set_p, op_sel_rom, op_test_s_eq_0, op_unknown      , op_unknown, op_unknown, op_clr_s, op_unknown,       op_test_p, op_c_to_data, op_del_sel_rom,op_data_to_c,op_unknown, 
	op_unknown, op_set_s, op_nop, op_set_p, op_sel_rom, op_test_s_eq_0, op_unknown      , op_unknown, op_unknown, op_clr_s, op_down_rotate,   op_test_p, op_unknown,   op_unknown,    op_unknown,  op_unknown, 
	op_unknown, op_set_s, op_nop, op_set_p, op_sel_rom, op_test_s_eq_0, op_unknown      , op_unknown, op_unknown, op_clr_s, op_unknown,       op_test_p, op_unknown,   op_del_sel_rom,op_unknown,  op_unknown, 
	op_unknown, op_set_s, op_nop, op_set_p, op_sel_rom, op_test_s_eq_0, op_unknown      , op_unknown, op_unknown, op_clr_s, op_clear_reg,     op_test_p, op_unknown,   op_unknown,    op_unknown,  op_unknown, 
	op_unknown, op_set_s, op_nop, op_set_p, op_sel_rom, op_test_s_eq_0, op_unknown      , op_unknown, op_unknown, op_clr_s, op_unknown,       op_test_p, op_unknown,   op_del_sel_rom,op_unknown,  op_unknown, 
};
#endif

static void (* const _op_fcn_classic   [16])() = {	op_nop, op_set_s, op_nop, op_set_p, op_sel_rom, op_test_s_eq_0,op_load_constant, 0, op_nop, op_clr_s, 0, op_test_p, op_nop, op_del_sel_rom, 0, 0 };
static void (* const _op_fcn_classic0a [8])() = {	op_display_toggle, op_m_exch_c, op_c_to_stack, op_stack_to_a, op_display_off, op_m_to_c, op_down_rotate, op_clear_reg };

// woodstock instruction set HP21-29,HP67
//static void (* const _op_fcn_0000 [256])() =
//{
//	op_nop,               op_set_s,  op_clear_reg,      op_clr_s, op_gokeys,            op_test_s_eq_1, op_load_constant, op_test_s_eq_0, op_sel_rom, op_test_p_eq, op_c_to_register, op_test_p_ne, op_unknown,         op_del_sel_rom,  op_register_to_c, op_set_p,
//	op_crc_test_motor_on, op_set_s,  op_clear_s,        op_clr_s, op_keys_to_a,         op_test_s_eq_1, op_load_constant, op_test_s_eq_0, op_sel_rom, op_test_p_eq, op_c_to_register, op_test_p_ne, op_unknown,         op_del_sel_rom,  op_register_to_c, op_set_p,
//	op_unknown,           op_set_s,  op_display_toggle, op_clr_s, op_a_to_rom_addr,     op_test_s_eq_1, op_load_constant, op_test_s_eq_0, op_sel_rom, op_test_p_eq, op_c_to_register, op_test_p_ne, op_crc_motor_on,    op_del_sel_rom,  op_register_to_c, op_set_p,     
//	op_crc_test_f1,       op_set_s,  op_display_off,    op_clr_s, op_display_reset_twf, op_test_s_eq_1, op_load_constant, op_test_s_eq_0, op_sel_rom, op_test_p_eq, op_c_to_register, op_test_p_ne, op_crc_motor_off,   op_del_sel_rom,  op_register_to_c, op_set_p,          
//	op_crc_set_f2,        op_set_s,  op_mx,             op_clr_s, op_binary,            op_test_s_eq_1, op_load_constant, op_test_s_eq_0, op_sel_rom, op_test_p_eq, op_c_to_register, op_test_p_ne, op_unknown,         op_del_sel_rom,  op_register_to_c, op_set_p, 
//	op_crc_test_f2,       op_set_s,  op_mx,             op_clr_s, op_circulate_a_left,  op_test_s_eq_1, op_load_constant, op_test_s_eq_0, op_sel_rom, op_test_p_eq, op_c_to_register, op_test_p_ne, op_crc_test_card_in,op_del_sel_rom,  op_register_to_c, op_set_p,           
//	op_crc_set_f3,        op_set_s,  op_mx,             op_clr_s, op_dec_p,             op_test_s_eq_1, op_load_constant, op_test_s_eq_0, op_sel_rom, op_test_p_eq, op_c_to_register, op_test_p_ne, op_unknown,         op_del_sel_rom,  op_register_to_c, op_set_p,
//	op_crc_test_f3,       op_set_s,  op_mx,             op_clr_s, op_inc_p,             op_test_s_eq_1, op_load_constant, op_test_s_eq_0, op_sel_rom, op_test_p_eq, op_c_to_register, op_test_p_ne, op_crc_test_prot,   op_del_sel_rom,  op_register_to_c, op_set_p,            
//	op_crc_set_f4,        op_set_s,  op_stack_to_a,     op_clr_s, op_return,            op_test_s_eq_1, op_load_constant, op_test_s_eq_0, op_sel_rom, op_test_p_eq, op_c_to_register, op_test_p_ne, op_bank_switch,     op_del_sel_rom,  op_register_to_c, op_set_p,
//	op_crc_test_f4,       op_set_s,  op_down_rotate,    op_clr_s, op_pik_home,          op_test_s_eq_1, op_load_constant, op_test_s_eq_0, op_sel_rom, op_test_p_eq, op_c_to_register, op_test_p_ne, op_c_to_addr,       op_del_sel_rom,  op_register_to_c, op_set_p, 
//	op_crc_set_f0,        op_set_s,  op_y_to_a,         op_clr_s, op_pik_cr,            op_test_s_eq_1, op_load_constant, op_test_s_eq_0, op_sel_rom, op_test_p_eq, op_c_to_register, op_test_p_ne, op_clear_data_regs, op_del_sel_rom,  op_register_to_c, op_set_p,
//	op_crc_clear_f0,      op_set_s,  op_c_to_stack,     op_clr_s, op_pik_keys,          op_test_s_eq_1, op_load_constant, op_test_s_eq_0, op_sel_rom, op_test_p_eq, op_c_to_register, op_test_p_ne, op_c_to_data,       op_del_sel_rom,  op_register_to_c, op_set_p,
//	op_crc_set_f1,        op_set_s,  op_decimal,        op_clr_s, op_unknown,           op_test_s_eq_1, op_load_constant, op_test_s_eq_0, op_sel_rom, op_test_p_eq, op_c_to_register, op_test_p_ne, op_rom_selftest,    op_del_sel_rom,  op_register_to_c, op_set_p,
//	op_crc_clear_f1,      op_set_s,  op_unknown,        op_clr_s, op_unknown,           op_test_s_eq_1, op_load_constant, op_test_s_eq_0, op_sel_rom, op_test_p_eq, op_c_to_register, op_test_p_ne, op_unknown,         op_del_sel_rom,  op_register_to_c, op_set_p,    
//	op_unknown,           op_set_s,  op_f_to_a,         op_clr_s, op_unknown,           op_test_s_eq_1, op_load_constant, op_test_s_eq_0, op_sel_rom, op_test_p_eq, op_c_to_register, op_test_p_ne, op_pik_print6,      op_del_sel_rom,  op_register_to_c, op_set_p,       
//	op_crc_write_prot,    op_set_s,  op_f_exch_a,       op_clr_s, op_pik_print3,        op_test_s_eq_1, op_load_constant, op_test_s_eq_0, op_sel_rom, op_test_p_eq, op_c_to_register, op_test_p_ne, op_nop,             op_del_sel_rom,  op_register_to_c, op_set_p,
//};

//static void (* const _op_fcn_0100 [4])() = {	op_set_s, op_test_s_eq_0, op_clr_s, op_del_sel_rom, };

//static void (* const _op_fcn_02xx [4])() = {	op_nop, op_load_constant, op_c_to_register, op_register_to_c, };     
//static void (* const _op_fcn_0300 [4])() = {	op_clr_s, op_test_s_eq_0, op_test_p_ne, op_set_p, };

// opcode function pointers for 0x000-0x3f0 for bit0-3=0
#ifdef HP67 // include CRC and PIK instructions
static void (* const _op_fcn_0000 [64])() =
{
	op_nop,               op_gokeys,            op_sel_rom, op_unknown,   
	op_crc_test_motor_on, op_keys_to_a,         op_sel_rom, op_unknown,   
	op_unknown,           op_a_to_rom_addr,     op_sel_rom, op_crc_motor_on,        
	op_crc_test_f1,       op_display_reset_twf, op_sel_rom, op_crc_motor_off,            
	op_crc_set_f2,        op_binary,            op_sel_rom, op_unknown,        
	op_crc_test_f2,       op_circulate_a_left,  op_sel_rom, op_crc_test_card_in,           
	op_crc_set_f3,        op_dec_p,             op_sel_rom, op_unknown,       
	op_crc_test_f3,       op_inc_p,             op_sel_rom, op_crc_test_prot,              
	op_crc_set_f4,        op_return,            op_sel_rom, op_bank_switch, 
	op_crc_test_f4,       op_nop,/*op_pik_home*/op_sel_rom, op_c_to_addr,        
	op_crc_set_f0,        op_nop,/*op_pik_cr*/  op_sel_rom, op_clear_data_regs, 
	op_crc_clear_f0,      op_nop,/*op_pik_keys*/op_sel_rom, op_c_to_data,   
	op_crc_set_f1,        op_unknown,           op_sel_rom, op_rom_selftest,  
	op_crc_clear_f1,      op_unknown,           op_sel_rom, op_unknown,             
	op_unknown,           op_unknown,           op_sel_rom, op_nop, /*op_pik_print6*/             
	op_crc_write_prot,    op_nop,/*op_pik_print3*/  op_sel_rom, op_nop,         
};
#else

static void (* const _op_fcn_0000 [13])() =
{
op_gokeys,op_keys_to_a,op_a_to_rom_addr,op_display_reset_twf,op_binary,op_circulate_a_left,op_dec_p,op_inc_p,op_return,op_c_to_addr,op_clear_data_regs, op_c_to_data,op_rom_selftest
};
#endif

static void (* const _op_fcn_0100 [4])() = {	op_set_s, op_test_s_eq_1, op_test_p_eq, op_del_sel_rom, };
static void (* const _op_fcn_02xx [4])() = {	op_nop, op_load_constant, op_c_to_register, op_register_to_c, };     
static void (* const _op_fcn_0300 [4])() = {	op_clr_s, op_test_s_eq_0, op_test_p_ne, op_set_p, };
static void (* const _op_fcn_0200 [16])() = {	
	op_clear_reg, op_clear_s, op_display_toggle, op_display_off, op_mx, op_mx, op_mx, op_mx,
	op_stack_to_a, op_down_rotate, op_y_to_a, op_c_to_stack, op_decimal, op_unknown, op_f_to_a, op_f_exch_a,
};
#endif
#endif


void act_execute_cycle()
{
  uint8_t op4,op2;

 	opcode = Getopcode();
  
#ifdef HP01
  if(HPType==0x01)
  {
    if(hp01flags & F_SLEEP)  // sleep mode will be ended by key press
      return;

    if (opcode == 0x190) // 0620 HP-01 SLEEP instruction ? original HP-.01 code has two SLEEP instructions, skip the first
    {
      if(act_pc==0321) // SLEEP not valid if followed by P=5 debounce routine
         opcode = 0; // nop
    }
  }
#endif

   act_flags &= ~AF_PREV_CARRY;
   if(act_flags & AF_CARRY)
      act_flags |= AF_PREV_CARRY; // set PREV_CARRY if CARRY was set
   act_flags &= ~AF_CARRY;

  act_pc++;
//  instructioncnt++;

  if(TypeFlags & TF_CLASSIC /*|| HPType==0x01*/)
  {
    if((uint8_t)act_pc==0)  // stay in same page
      act_pc=act_del_rom<<8;
#if 0
    if(act_flags & AF_KEYDOWN) // HP-45 needs keydown for more than 1 cycle debounce
    {
      act_s|=1<<0;
      if(--keydowncnt==0)
         act_flags&=~AF_KEYDOWN;
    }
#endif
  }

  switch ((char)opcode & 3)
	{
		case 0:
        op2=opcode>>2; // index to function tables
        op6=op2>>4;    // used by op_xxx() functions
        op4=opcode >> 4 & 3;  // needed for optimize for size

#ifdef HP01
        if(HPType==0x01)
					_op_fcn_0000 [(op2)] ();
        else
#endif
         if(TypeFlags & TF_CLASSIC)
         {
						if(op2==0x07)
  						op_dec_p();
						else if(op2==0x0f)
  						op_inc_p();
						else if(op2==0x34)
  						op_gokeys();
						else if(op2==0x0c)
  						op_return();
						else if(op2==0x0d)
  						op_clear_s();
						else if(op2==0x8d || op2==0xad)
  						op_del_sel_grp();
						else if(op2==0x9c)
  						op_c_to_addr();
						else if(op2 ==0xbc)
  						op_c_to_data();
						else if(op2 ==0xbe)
  						op_data_to_c();
						else if(_op_fcn_classic[op2 & 0x0f])
  						_op_fcn_classic[op2 & 0x0f]();
						else if((op2 & 0x0f) == 0x0a)
  						_op_fcn_classic0a[op6>>1]();
          }
          else
          {
            varbank0=opcode & 0x0c;
            if(varbank0==0)  // switch ((char)opcode & 0x0c)
            {
#ifdef HP67
                _op_fcn_0000 [opcode>>4] ();
#else
              if(opcode==01060)
                op_bank_switch();
              else 
              if(op4==2) // HP-25/29 has reduced instruction set
                op_sel_rom();
              else if((op4 & 1) && op6<13) // kein grosser Zeitnachteil, da relativ seltene Befehle
                _op_fcn_0000 [op6] (); // op6 nie >12 bei HP-25,HP-29 op4==1 -> op6 0-8 op4==3 -> op6 9-11 
#endif
            } else if(varbank0==0x04)  
            {
					   _op_fcn_0100 [op4] ();
            } else if(varbank0==0x08)  
            {
						  if (op4)
							  _op_fcn_02xx [op4] ();
						  else
							  _op_fcn_0200 [op6] (); 
            } else if(varbank0==0x0c)  
            {
					   _op_fcn_0300 [op4] ();
            } 
#ifdef USEASM
            if(op2==0x04 || op2==0x14 || op2==0x24) // op_gokeys, keys_to_a, a_to_rom address 
              CheckAddress(); // assembler cant call external function ???
#endif
          }
					break;
				case 1:
          op_jsb();
          break;
				case 2:
          op_arith();
          break;
				case 3:
          op_goto();
          break;
		}

  	if(opcode != 0x1d0) // not op_inc_p for label search
	  	act_flags&=~AF_PCARRY;

    if(act_inst_state==branch)
    {
			act_inst_state = norm;
      if(!(TypeFlags & TF_CLASSIC)) // classic doesn't have 10-bit branch instruction
      {
  			if ( !(act_flags & AF_CARRY))
        {
          ReadInternalROM(); // read jump address from act_pc
			  	act_pc = (act_pc & 0xFC00) | opcode;
        } else
        {
          act_flags &= ~AF_CARRY;
          act_pc++; // 12-bit Woodstock increment
        }
      }
    }
#if 0 // SPICE selftest instruction
    else if(act_inst_state==selftest)
    {
			if (opcode == 01060)
              op_bank_switch(); 
			if ((act_pc & 0x3FF)==0)
              rom_selftest_done();
    } 
#endif
}

void act_press_key (uint8_t keycode)
{
#ifdef HP01
  if(HPType==0x01)
  {
    if(keycode!=63) // dont set s(0) if wakeup otherwise endless loop
	    act_s |= 1<<0;
    hp01flags&=~F_SLEEP;
  }
  else
#endif
  if(TypeFlags & TF_CLASSIC)
  {
    if(!(act_flags & AF_KEYDOWN))
    {
	    act_s |= 1<<0;
      act_flags|=AF_KEYDOWN;
//    keydowncnt=10;
    }
  }
  else
  {
	  act_s |= 1<<15;
  }
	act_key_buf= keycode;
}

void act_release_key()
{
  act_flags&=~AF_KEYDOWN; //  act_key_buf darf nicht auf 0 gesetzt werden, sonst wird bei HP-25 falsch gesprungen
#ifdef HP01
   if(HPType==0x01)
  	act_s &= ~(1<<0);
//   else
#endif
//	act_s &= ~(1<<15);  // no action allowed because flag will be reset by firmware
}

void act_reset()
{
	base=10;
	act_flags = AF_RUNMODE; // default mode
	crc_flags=0;
	act_del_rom=0;
	act_inst_state = norm;
	act_sp = 0;
	act_key_buf = 0;  // no key has been pressed
//  act_switch=4;  // 1 = trace mode 2=norm mode  4=manual
	act_pc = 0;
	act_p = 0;
	act_s=0;
	act_f=0;
	opcode=0;
  act_ram_size= HPType==0x55 ? 30 : 0;  // HP35=0, HP-45=0,HP-80=0 HP-55=30
	op_clear_reg();
	act_stack[0]=0;
	act_stack[1]=0;

// hp01flags=0;  // cleared only by power on reset
}

#if 0
// not used to save memory space
void act_clear_memory()
{
	uint8_t i;
	for (i = 0; i < RAMBANKS; i++) // 2-8 RAM banks
	{
    getrampt(i*RAMBANKSIZE); // dest=act_ram[i]
 		first=0; last=RAMBANKSIZE*7-1; // last is 55 reg_zero() does not need 16-bit increment
 		reg_zero();
	}
}
void act_clear()
{
  act_reset();
  act_clear_memory();
}

#endif

#if 0
// this version uses more program space than initializing variables separately
void act_reset()
{
  uint8_t *pt=(uint8_t *)ACT_STATE;
  for(uint16_t i=0;i<32+80+448;i++) // clear 32 byte processor context + 80 byte registers + 448 ram bytes
    pt[i]=0;
	base=10;
  act_switch=4;  // 1 = trace mode 2=norm mode  4=manual
  act_ram_size= HPType==0x55 ? 30 : 0;  // HP35=0, HP-45=0,HP-80=0 HP-55=30
}
#endif