#include "stdint.h"
#include "stdbool.h"

#define STACK_SIZE 2  // ACT stack depth two 12-bit addresses
#define WSIZE   14   // register size 14 BCD digits 7 byte
#define WSIZE01 12  // HP-01 register size 12 BCD digits 6 byte
#define EXPSIZE 3  // two exponent and one exponent sign digit

typedef enum {
	norm,
	branch,
	selftest
} inst_state_t;

typedef uint8_t reg_t [WSIZE];
typedef uint8_t creg_t [WSIZE/2];

// act_flags don't change any bits because of assembler routines
#define AF_RUNMODE	  (1<<0)
#define AF_CARRY			(1<<1)
#define AF_PREV_CARRY (1<<2)
#define AF_DEL_ROM		(1<<3)
#define AF_KEYDOWN    (1<<4)
#define AF_DISPLAY_ON (1<<5)
#define AF_PCARRY		  (1<<6)
#define AF_BANK			  (1<<7)

// hp01flags don't change any bits because of assembler routines
#define F_SLEEP       (1<<0)
#define F_BLINK       (1<<1)
#define F_SCWP        (1<<2)
#define F_SWDEC       (1<<3)
#define F_SWSTARTED   (1<<4)
#define F_ALARMACTIV  (1<<5)
#define F_WAKEUP      (1<<6)
#define F_CONTMEMORY  (1<<7)

// systemflags
#define SF_BPENABLE      (1<<0)
#define SF_CODEBPENABLE  (1<<1)
#define SF_BPTRIGGERED   (1<<2)
#define SF_SLIDERCHANGED (1<<3)

// absolute addresses for asm code

// bank 0
// variables below 0x50 are not part of context save
extern uint8_t systemflags @ 0x4d;
extern uint8_t hp01flags @ 0x4e;
extern uint8_t varbank0 @ 0x4f; // variable located in ram bank 0
extern uint8_t trigmode @ 0x4f; // variable located in ram bank 0
// ACT_STATE context begins at 0x50 until 0x6f, 0x50 to 0x5b are not necessary for context save
// adresses of ACT_STATE variables may not contain gaps until ACT_RAM
// because they are saved and restored when calculator context switch
extern uint8_t *src @ 0x50;
extern uint8_t srch @0x51;
extern uint8_t *dest @ 0x52;
extern uint8_t desth @0x53;
extern uint8_t *src2 @ 0x54;
extern uint8_t src2h @0x55;
extern uint8_t first @ 0x56 ;
extern uint8_t last @ 0x57;
extern uint8_t s2 @ 0x58;
extern uint8_t op6 @ 0x59;
extern uint16_t opcode @ 0x5a;
extern uint8_t opcodel @ 0x5a;
extern uint8_t opcodeh @ 0x5b;
extern uint8_t ProgramNr @ 0x5c;
extern uint8_t act_flags @ 0x5d;
extern uint8_t crc_flags @ 0x5e;
extern uint8_t base @ 0x5f;
extern uint16_t act_pc @ 0x60; // program counter
extern uint8_t act_pcl @ 0x60; // program counter
extern uint8_t act_del_rom @ 0x62; // delayed rom instruction
extern uint8_t act_sp @ 0x63; // stack pointer
extern uint8_t act_p @ 0x64;
extern inst_state_t act_inst_state @ 0x65;
extern uint16_t act_s @ 0x66;
extern uint8_t act_f @ 0x68;
extern uint8_t act_key_buf @ 0x69;
extern uint8_t act_ram_addr @ 0x6a;
extern uint8_t act_ram_size @ 0x6b;
extern uint16_t act_stack[STACK_SIZE] @ 0x6c; // two level stack until 0x6f

// bank 3
// ACT_REGS
// must be contigous without gaps for asm code and context save/restore
extern reg_t  act_a @ 0x0a0;
extern reg_t  act_b @ 0x0ae;
extern reg_t  act_c @ 0x0bc;
extern creg_t act_y @ 0x0ca;
extern creg_t act_z @ 0x0d1;
extern creg_t act_t @ 0x0d8;
extern creg_t act_n @ 0x0df;
extern creg_t act_m @ 0x0e6;
extern uint8_t reserved_ed @ 0x0ed;
extern uint8_t DebugCommand @ 0x0ee;

extern creg_t act_ram1[8] @ 0x120; // alternating access to first 8 ACT_RAM registers
extern uint8_t act_bank2[80] @ 0x120; 
extern uint8_t act_bank3[80] @ 0x1a0; 
extern uint8_t act_bank4[80] @ 0x220; 
extern uint8_t act_bank5[80] @ 0x2a0; 
extern uint8_t act_bank6[80] @ 0x320; 
extern uint8_t act_bank7[48] @ 0x3a0;
extern reg_t act_reg @ 0x3d0;
extern reg_t act_dsp @ 0x3de;
extern uint16_t act_breakpoint @ 0x3ec;
extern uint16_t codebreakpoint @ 0x3ee;

extern reg_t act_cl @ 0x05b8;
extern reg_t act_sw @ 0x05d4;
extern reg_t act_al @ 0x05c6;
extern uint8_t act_weekday @ 0x0566; // 0-6 = sunday to saturday

// linear ACT_RAM memory 8*56=448 bytes 

#define ACT_STATE 0x2030 // linear address of act context   0x050
#define ACT_REGS  0x2050 // linear address of act registers 0x0a0
#define ACT_RAM   0x20a0 // linear address of act_ram       0x120
//extern creg_t act_ram[] @ 0x20a0; // linear address of act_ram[] 0x2000-0x29AF, will not compile correctly because calculates 0x!20


bool act_execute_instruction();
void act_execute_cycle();
void act_reset();
void act_clear();

void act_press_key (uint8_t keycode);
void act_release_key();

void getrampt(uint8_t addr);
void getrampt1(uint8_t addr);
void reg_decompress();
void reg_compress();
void reg_normalize(uint8_t *dest);
void register_to_c(uint8_t addr);
void c_to_register(uint8_t addr);
void register_to_m(uint8_t addr);
void m_to_register(uint8_t addr);
void setfield(uint8_t field);
void reg_zero();
void reg_copy();
void reg_exch();
void reg_shift_right();
void reg_shift_left();
void reg_inc();
void reg_add();
void reg_sub();
void reg_test_equal();
void reg_test_nonequal();
void handle_del_rom();

