;/***********************************************************************************/
;/* function:   BootLoader ldt ein neues Programm in das Flash Memory              */
;/*---------------------------------------------------------------------------------*/
;/*---------------------------------------------------------------------------------*/
;/* purpose:    Dieser Programmteil wird nach Programmstart angesprungen und prft  */
;/* ob ein break an der seriellen Schnittstelle anliegt                             */
;/* wenn ja, wird auf ein update gewartet                                           */
;/* Die Baudrate ist fest auf 115200 Baud eingestellt.                              */
;/*---------------------------------------------------------------------------------*/
;/* history:    29-dez-14, V1.00, BE(PANAMATIK), adaption for PIC16F1518 ACT        */
;/*             31-jan-15, V1.00, BE(PANAMATIK), decryption                         */
;/*             05-jun-15, V1.04, BE(PANAMATIK), reserved boot area, version number */
;/*             21-oct-17, V1.11, BE(PANAMATIK), adaptiert fr PIC16F1519 HPLP      */
;/***********************************************************************************/
#include "pic16f1519.inc"

processor	16F1519 

; Variables
RXBYTE		equ	0x20 ; // received byte
CHKSUM		equ 0x21 ; // checksum accumulator
COUNTER		equ	0x22 ; // received byte counter
COUNTER1	equ	0x23 ; // general counter
COUNTER2	equ	0x24 ; // general counter
BLINKCNT1	equ	0x23 ; // LED blink counter
BLINKCNT2	equ	0x24 ; // LED blink counter
KRYPTOVALUE equ 0x25 ; // decryption value
KRYPTOVAL1  equ 0x29 ; // decryption value
DATA_BUFF	equ 0x30 ; // start of receive buffer

; defines

global	_StartOfLine

PACKETSIZE	equ	0x40 ; // 64 bytes/32 words) + address + checksum

; The bootloader must reside in Adress 0x000-0x00ff

   	psect	bootarea1,global, abs, ovrld,reloc=16,class=CODE,merge=1,delta=2
	ORG 0002h  ; reserved boot area 0000-00ff
	dw	0100h  ; Bootlader Version 1.00 and prevent C compiler from using this location
	dw	0

	psect	init,class=CODE,delta=2

; this entry point will be located at 0x000A directly behind start code in psect init

Bootloader:

	BANKSEL	(PORTA) 		; bank 0, BSR is 0 after Reset, but could be startcode before
	movlw	0FFH
	movwf	PORTA           ; /CE High /TS High, rows 1-6 high
	clrf	PORTB
	clrf	PORTC
	clrf	PORTD           ; 8 segements not selected
    movlw   3
	movwf	PORTE           ; CE1,CE2 high GPS disable
	BANKSEL	(OSCCON)          ; bank 1, same as TRISA-E 
	movlw	7AH               ; internal OSC 16 MHz, necessary for 115200 Baud
	movwf	BANKMASK(OSCCON)
	movlw	038H              ; RB0-7 input 
	movwf	BANKMASK(TRISA)   ; RA0-7 output
	movlw	0FFH              ; RB0-7 input 
	movwf	BANKMASK(TRISB)
	movlw	091H              ; all output except RX, SDO, CLKOUT
	movwf	BANKMASK(TRISC)
    clrf	BANKMASK(TRISD)   ; all output, 8 segments a-g d.p.
    clrf	BANKMASK(TRISE)   ; all output row 7-9
	movlw	05H
	movwf	BANKMASK(OPTION_REG) ; TMR0 internal clock FOSC/4, Prescaler assigned to Timer, rate 1:64, pull ups enabled
	BANKSEL	(ANSELB)
	clrf	BANKMASK(ANSELB); // all digital inputs
	clrf	BANKMASK(ANSELC); // all digital inputs, RX

	BANKSEL(COUNTER)
	clrf	COUNTER

WaitStart:
	decf	COUNTER ; wait until RX is high by weak pullup 256*4*250ns = 256 us
	btfss   3,2
	goto 	WaitStart
	btfsc	PORTC,7 ; // if(!PORTCbits.RC7) // receiving BREAK ?
	goto	0100h ;   // no, jump to application reset vector if RX is high
	
; RX is low, wait until RX goes high
WaitBreak:
;	btfss	PORTC,7 ; // while(!PORTCbits.RC7) // wait for end of break
;	goto	WaitBreak
;	nop
	
InitUart:
	BANKSEL(TXSTA)
	movlw	24h ;	TXSTA=0x24;   // 8-bit Transmission, Transmit enabled, asynchronous mode,BRGH=1 High Speed
	movwf	BANKMASK(TXSTA)
	movlw	08h ; BAUDCON=0x08; // BRG16=1
	movwf	BANKMASK(BAUDCON)
	movlw	23h-1 ; SPBRG=35-1;   // 115200 Baud 16 MHz BRG16=1 BRGH=1  SPBRG=16MHz/4/Baudrate-1 rounded up
	movwf	BANKMASK(SP1BRGL)
	movlw	80h ; 	RCSTA=0x90;   // serialPort enabled,8-bit receiption, enable continous receive, CREN=1
	movwf	BANKMASK(RCSTA)
	movf	BANKMASK(RCREG) ; clear RCIF
	bsf		BANKMASK(RCSTA),4

_StartOfLine:
    movlb   0   ; can be jumped from C code
    bcf     INTCON,7 ; GIE=0 
	clrf	PORTD           ; 8 segements not selected

StartOfLine:
	call	RdRS232 ; Get the data wait for : semicolon
	movf	RXBYTE,w
	xorlw	3ah       ;  Check for : start of Line character
	btfss	3,2  ; bz
	goto	StartOfLine ; Otherwise go back for another character

; semicolon received
	movlw	20h
	BANKSEL	(LATB)
	xorwf	BANKMASK(LATB)

	BANKSEL	(0)
	movlw	low DATA_BUFF ; Point to the buffer
	movwf	FSR0L
	clrf	FSR0H
	clrf	CHKSUM ; Reset checksum
	clrf	COUNTER ; Reset buffer count
GetNextDat:
;	movlw	0DFH  ; LED on while data receiving
;	movwf	PORTB

	call	RdRS232 ; Get the data
	addwf	CHKSUM ; Get sum
	MOVWI	FSR0++  ; Store into buffer
	incf	COUNTER
	movlw	PACKETSIZE+3
	xorwf	COUNTER,w
	btfss	3,2 ;
	goto	GetNextDat

; 2 address bytes, 64 Data bytes, 1 bytes checksum received

CheckSum:
	movf	CHKSUM,w ; Checksum test, must be 0
	btfss	3,2 ; bnz
	goto	SendNack ; wrong checksum

	comf	DATA_BUFF,w ; address 0xFF ? receive KryptoValue
	btfss	3,2
	goto 	WritePage

; Read Kryptovalue
	movf	DATA_BUFF+2,w ; get 4 byte int32 KryptoValue
	movwf	KRYPTOVALUE	
	movf	DATA_BUFF+3,w
	movwf	KRYPTOVALUE+1	
	movf	DATA_BUFF+4,w
	movwf	KRYPTOVALUE+2	
	movf	DATA_BUFF+5,w
	movwf	KRYPTOVALUE+3
	goto 	SendAck	

; decrypt and write 64 bytes into Flash
WritePage:
;	movlw	0BFH  ; LED off while flashing
;	movwf	PORTB
	movlw	low DATA_BUFF+2 ; Point to data in data buffer 
	movwf	FSR0L
	clrf	FSR0H
	call	Decrypt

	movf	DATA_BUFF,w  ; get Address MSB
	btfsc	3,2  ; address 0x00xx  dont write into bootarea
	goto	SendAck
	
	movlw	low DATA_BUFF+2
	movwf	FSR0L
	 BANKSEL	(DATA_BUFF)
	movf	DATA_BUFF,w  ; get Address MSB
	BANKSEL	(PMADRH)
	movwf	BANKMASK(PMADRH)
	BANKSEL	(DATA_BUFF)
	movf	DATA_BUFF+1,w  ; get Address LSB
	BANKSEL	(PMADRL)
	movwf	BANKMASK(PMADRL)

	call	FlashErase  ; erase page 32 14-bit words = 64 byte buffer
	call	FlashWrite

SendAck:
	movlw	06h ; // Send Ack
	goto	SendByte
SendNack:
	movlw	15h ; // Send Nack
SendByte:	
	BANKSEL(TXREG)
	movwf	BANKMASK(TXREG)
	goto	StartOfLine

; wait for Receive byte while LED blinks
RdRS232:
	BANKSEL(RCSTA)
	btfsc	BANKMASK(RCSTA), 1  ;OERR ; Reset on overrun
	reset
WaitforData:
	clrwdt
	call	blink
	BANKSEL(PIR1)
	btfss	PIR1, 5 ; RCIF ; Wait for data from RS232
	goto	WaitforData
ReadData:
	BANKSEL	(RCREG)
	movf	BANKMASK(RCREG), W ; Save the data
	BANKSEL	(RXBYTE)
	movwf	RXBYTE
	return

blink:
    BANKSEL (LATA)
    incf    BANKMASK(LATA)
	BANKSEL	(BLINKCNT1)
	decf	BLINKCNT1
	btfss	3,2
	return
	decf	BLINKCNT2
	btfss	3,2
	return
	movlw	07Fh
	BANKSEL	(LATD)
	xorwf	BANKMASK(LATD)
	return

;// erase one row 64 bytes of flash memory specified by PMADR
FlashErase:
	BANKSEL(PMCON1)
	BCF BANKMASK(PMCON1),6 ; CFGS ; Not configuration space
	BSF BANKMASK(PMCON1),4 ; FREE ; Specify an erase operation
	BSF BANKMASK(PMCON1),2 ; WREN ; Enable writes
	MOVLW 55h ; Start of required sequence to initiate erase
	MOVWF BANKMASK(PMCON2) ; Write 55h
	MOVLW 0AAh ;
	MOVWF BANKMASK(PMCON2) ; Write AAh
	BSF BANKMASK(PMCON1), 1; ,WR ; Set WR bit to begin erase
	NOP ; NOP instructions are forced as processor starts
	NOP ; row erase of program memory.
;
; The processor stalls until the erase process is complete
; after erase processor continues with 3rd instruction
	BCF BANKMASK(PMCON1),2 ; WREN ; Disable writes
	return

FlashWrite:
	BANKSEL(PMCON1)
	BCF BANKMASK(PMCON1),6 ; CFGS ; Not configuration space
	BCF BANKMASK(PMCON1),4 ; FREE ; Write
	BSF BANKMASK(PMCON1),5 ; LWLO ; Only Load Write Latches
	BSF BANKMASK(PMCON1),2 ; WREN ; Enable writes
	
LOOP
	MOVIW FSR0++ ; Load first data byte into lower
	MOVWF BANKMASK(PMDATL)
	MOVIW FSR0++ ; Load second data byte into upper
	MOVWF BANKMASK(PMDATH)
	MOVF BANKMASK(PMADRL),W ; Check if lower bits of address are '00000'
	XORLW 0x1F ; Check if we're on the last of 32 addresses
	ANDLW 0x1F ;
	BTFSC STATUS,2 ; Z ; Exit if last of 32 words,
	GOTO START_WRITE ;
	MOVLW 55h ; Start of required write sequence:
	MOVWF BANKMASK(PMCON2) ; Write 55h
	MOVLW 0AAh ;
	MOVWF BANKMASK(PMCON2) ; Write AAh
	BSF BANKMASK(PMCON1),1 ; WR ; Set WR bit to begin write
	NOP ; NOP instructions are forced as processor loads program memory write latches
	NOP ;
	INCF BANKMASK(PMADRL),F ; Still loading latches Increment address
	GOTO LOOP ; Write next latches

START_WRITE
	BCF BANKMASK(PMCON1),5 ; LWLO ; No more loading latches - Actually start Flash program memory write
	MOVLW 55h ; Start of required write sequence:
	MOVWF BANKMASK(PMCON2) ; Write 55h
	MOVLW 0AAh ;
	MOVWF BANKMASK(PMCON2) ; Write AAh
	BSF BANKMASK(PMCON1),1 ; WR ; Set WR bit to begin write
	NOP ; NOP instructions are forced as processor writes all the program memory write latches simultaneously
	NOP ; to program memory.
; After NOPs, the processor stalls until the self-write process in complete
; after write processor continues with 3rd instruction
	BCF BANKMASK(PMCON1),2 ; WREN ; Disable writes
	return

;void xorshift32()
;{
;  KryptoValue ^= KryptoValue << 13;
;  KryptoValue ^= KryptoValue >> 17;
;  KryptoValue ^= KryptoValue << 5;
;}

copykryptovalue:
	movf	KRYPTOVALUE,w
	movwf	KRYPTOVAL1
	movf	KRYPTOVALUE+1,w
	movwf	KRYPTOVAL1+1
	movf	KRYPTOVALUE+2,w
	movwf	KRYPTOVAL1+2
	movf	KRYPTOVALUE+3,w
	movwf	KRYPTOVAL1+3
	return

xorkryptovalue:
	movf	KRYPTOVAL1,w
	xorwf	KRYPTOVALUE
	movf	KRYPTOVAL1+1,w
	xorwf	KRYPTOVALUE+1
	movf	KRYPTOVAL1+2,w
	xorwf	KRYPTOVALUE+2
	movf	KRYPTOVAL1+3,w
	xorwf	KRYPTOVALUE+3
	return
	
xorshift32:
	movlw		13
	movwf	COUNTER2
	call 	copykryptovalue
xorshift1:
	lslf    KRYPTOVALUE
	rlf     KRYPTOVALUE+1
	rlf     KRYPTOVALUE+2
	rlf     KRYPTOVALUE+3
	decfsz  COUNTER2
	goto    xorshift1
	call	xorkryptovalue

	movlw		17
	movwf	COUNTER2
	call 	copykryptovalue
xorshift2:
	lsrf    KRYPTOVALUE+3
	rrf     KRYPTOVALUE+2
	rrf     KRYPTOVALUE+1
	rrf     KRYPTOVALUE
	decfsz  COUNTER2
	goto 	xorshift2
	call	xorkryptovalue

	movlw		5
	movwf	COUNTER2
	call 	copykryptovalue
xorshift3:
	lslf    KRYPTOVALUE
	rlf     KRYPTOVALUE+1
	rlf     KRYPTOVALUE+2
	rlf     KRYPTOVALUE+3
	decfsz  COUNTER2
	goto 	xorshift3
	call	xorkryptovalue
	return

Decrypt:
 	movlw	16                    ; decrypt 16*4 byte
 	movwf	COUNTER1
Decrypt1:
 	movf	KRYPTOVALUE,w
 	XORWF	INDF0
 	INCF	FSR0L
 	movf	KRYPTOVALUE+1,w
 	XORWF	INDF0
 	INCF	FSR0L
 	movf	KRYPTOVALUE+2,w
 	XORWF	INDF0
 	INCF	FSR0L
 	movf	KRYPTOVALUE+3,w
 	XORWF	INDF0
 	INCF	FSR0L
 	call	xorshift32
 	decfsz	COUNTER1
 	goto Decrypt1
	return
	
	dw		00beh

	ORG 0100h  ; reserved boot area 0000-00ff

endasm
