/*
 *  hp82240.c
 *  
 *
 *  Created by Bernhard Emese on 18.04.15.
 *  Copyright 2015 Panamatik. All rights reserved.
 *
 */
#include "xc.h"
#include "hp25lp.h"
#include "act.h"
#include "display.h"
#include "sst25pf040.h"
#include "hp82240.h"
#include "mnemonic.h"

#ifdef HP82240

#define IRCLKPERIOD (_XTAL_FREQ/4/32768)  // 32768 Hz = 30 us Period

#define LF 0x0a

static void InitHP82240()
{
	StopDisplay(); // stop display interrupt because PWM2 uses the same timer TMR2

// use CCP2 PWM for 32768 Hz IR pulses
// PWM2 Output Low Level Idle Mode

	PR2=IRCLKPERIOD-1; // PWM Period 32,768 kHz
                 
// Set TMR2 Prescale Value 1:1 PostScale 1:1 Timer2 on

	CCP2CON=0x0c; // set PWM mode, reset lower 2 bits of PWM duty cycle
	CCPR2L=0;          // PWM DutyCycle 0%
}

static void ExitHP82240()
{
	CCP2CON=0x00; // set PWM mode, reset lower 2 bits of PWM duty cycle
	StartDisplay();
}

// send 1/2 bit, 14 cycles of 30 us,
// send burst of 8 pulses of 15 us each if Bit is set
static void SendHP82240Burst(uint8_t Bit)
{
	if(Bit & 1)
	  CCPR2L=IRCLKPERIOD/2; // set duty cycle 50% if high bit, otherwise is 0%

    TMR2IF=0;  // reset Timer2 to PR2 interrupt flag

	for(uint8_t i=0;i<14;i++) // half bit time is 14 cycles
	{
		while(TMR2IF==0) // wait 30 us until PWM cycle is finished
		  ;
		TMR2IF=0;

		if(i==7)    // stop pulse output after 8 pulses
		  CCPR2L=0; // set duty cycle 0% low level
  }
}

static void SendHP82240Bit(uint8_t Bit)
{
// one bit is high burst followed by low burst, else exchanged
   SendHP82240Burst(Bit);	
   SendHP82240Burst(Bit^1);	
}

// table for fast calculation of error correction bits
static const uint8_t correctiontab[8] = { 0x03, 0x05, 0x06, 0x09, 0x0a, 0x0c, 0x0e, 0x07};
 
// send one frame which is one ASCII character with start and error correction bits
// a frame is 30 cycles, 
static void SendHP82240Frame(uint8_t byte)
{
uint8_t byte1=byte;
uint8_t b=0;
uint8_t i=0;
uint16_t irbyte;

// calculate correction bits 9-11
	while(byte1)
	{
		if(byte1 & 1) b^=correctiontab[i];
		i++;
		byte1>>=1;	
	}

// equivalent enrolled calculation
//	if(byte1 & (1<<0) b^=0x03;
//  if(byte1 & (1<<1) b^=0x05;
//	if(byte1 & (1<<2) b^=0x06;
//	if(byte1 & (1<<3) b^=0x09;
//	if(byte1 & (1<<4) b^=0x0a;
//	if(byte1 & (1<<5) b^=0x0c;
//	if(byte1 & (1<<6) b^=0x0e;
//	if(byte1 & (1<<7) b^=0x07;

	irbyte=((uint16_t)b<<8) | byte;

	TMR2=0; // reset Timer 2 to zero
	TMR2ON=1; // start PWM Timer from 0
		
	for(i=0;i<3;i++)
		SendHP82240Burst(1); // send 3 start bits
			 
	for(i=0;i<12;i++) // send 12 bits, 4 bit correction, 8 bit data
	{
		SendHP82240Bit((irbyte & 0x800)!=0); // MSB first
		irbyte<<=1;
	}
		
	for(i=0;i<3;i++)
		SendHP82240Burst(0); // send 3 stop bits

	TMR2ON=0;
}

static const uint8_t AlphaDigits[6] = { 'r','F','o', 'P', 'E', ' '};

// send actually displayed X register
static void SendHP82240Digits()
{
	for(uint8_t n=0;n<MAXDIGITS;n++)
	{
		uint8_t digit,a;

    digit=act_dsp[WSIZE-1-n]; //returns 0-15 or ' ' or '-',  bit 7 set if additional decimal point 

    a=digit & 0x7f;
    if(a<=9)         // numeric digit 0-9
	    a|='0'; // -> ASCII digit 0-9 
    else if(a<=15) // alpha digits or blank
		  a=AlphaDigits[a-10];

		if(a!=' ' || !(digit & 0x80)) // don't print space when only decimal point, can occur in program step
		  SendHP82240Frame(a);
		if(digit & 0x80)  // decimal point ?
		  SendHP82240Frame('.'); // add decimal point
  } 
}

void SendHP82240LineFeed()
{
	InitHP82240();
	SendHP82240Frame(LF); // Line Feed
	ExitHP82240();
}

// send actual display register act_a, act_b, send 12 digits with comma, each needs 12,4 ms  = ca. 148 ms
void SendHP82240Display()
{
	InitHP82240();
	SendHP82240Digits(); // show actual Display
	SendHP82240Frame(LF); // Line Feed
	ExitHP82240();
}

void SendHP82240mnemonic(uint8_t n)
{
	uint8_t a,i;

  n=SetMnemonic(n,MnemonicAddr); // read into act_reg+2
  for(i=1;i<n;i++)
  {
    a=act_reg[WSIZE-1-i];
	  SendHP82240Frame(a); // print mnemonic
  }
}

void SendHP82240ProgramStep(uint8_t code,uint8_t mode)
{
	InitHP82240();

	if(mode & 1)
	{
		SendHP82240Digits(); // show actual program step display  i.e. "01      13 00"
		SendHP82240Frame(' '); // space
	}
	SendHP82240mnemonic(code);
	SendHP82240Frame(LF); // Line Feed
	ExitHP82240();
}

#endif