/*
 *  nmea.c
 *  
 *  nmea module for HP-25E
 *
 *  Created by Bernhard Emese on 07.02.16 - 26.3.2016.
 *  Copyright 2016 Panamatik. All rights reserved.
 *
 * NMEA Modul parses incoming serial NMEA data from GPS modul and checks incoming GPGGA and GPVTG Messages
 * records geographic position, latitude, longitude, speed, track, altitude etc.
 * automatically stores data into HP-25E RAM registers if requested for calculations
 */

#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include "xc.h"
#include "hp25lp.h"
#include "act.h"
#include "serial.h"
#include "keyboard.h"
#include "pcf2127.h"
#include "nmea.h"

#ifdef NMEAMENU

static const uint8_t GPGGA[5] = "GPGGA";  // compare string for message
static const uint8_t GPVTG[5] = "GPVTG";  // compare string for message

// buffers for holding last received values, ASCII numbers with optional decimal, null terminated

struct sNMEA
{
char TagName[6];           // GPGGA or GPVTG
char UTCTime[7];           // Time 000000 HHMMSS
char Latitude[11];         // Latitude 00000.0000
char NS[2];                // North South
char Longitude[11];        // Longitude 00000.0000
char EW[2];                // East West
char Valid[2];             // Data valid
char Satellites[3];        // number of satellites
char HDOP[7];              // Horizontal Dilution Of Precision
char Altitude[7];          // Altitude 000.00
char Track[7];             // Track  000.00
char Speed[7];             // Speed 000.00 knots or km/h
};
// 72 bytes

struct sNMEA NMEA @ 0x420; // sizeof(struct sNMEA) = 72 bytes, max 80 allowed, share with rombuf

static const char RxBytesTab[] = {
sizeof(NMEA.TagName),
sizeof(NMEA.UTCTime),
sizeof(NMEA.Latitude),
sizeof(NMEA.NS),
sizeof(NMEA.Longitude),
sizeof(NMEA.EW),
sizeof(NMEA.Valid),
sizeof(NMEA.Satellites),
sizeof(NMEA.HDOP),
sizeof(NMEA.Altitude),
sizeof(NMEA.Track), // true Track
0,0,0,              // skip ...,T,magnetic Track,M,...
sizeof(NMEA.Speed), // speed knots
0,                  // N, for knots
sizeof(NMEA.Speed)  // Speed km/h
};

// No of data words
#define NROFNMEADATA  8

// Offsets buffers UTC, Latitude, Longitude etc. in order of NMEAShow
static const char NMEAptTable[NROFNMEADATA] = {
6,  // offsetof(struct sNMEA,UTCTime),    // 6
13, //offsetof(struct sNMEA,Latitude),   // 13 
26, //offsetof(struct sNMEA,Longitude),  // 26
65, //offsetof(struct sNMEA,Speed),      // 65, knots or km/h
58, //offsetof(struct sNMEA,Track),      // 58
51, //offsetof(struct sNMEA,Altitude),   // 51
44, //offsetof(struct sNMEA,HDOP),       // 44
41, //offsetof(struct sNMEA,Satellites), // 41
};

// any of the values can be automatically stored into a destination RAM register if RegisterNr>0
static uint8_t  RegisterNr[NROFNMEADATA]; // registers for automatic update 0=no update

uint8_t NMEAFlags;        // see Flags definitions

static uint8_t  NMEAShow;         // menu number, 0=Time 1= Latitude 2=Longitude 3=Speed 4=Track 5=Altitude 6= HDOP 7= Satellites
static uint8_t  RxBytes;          // Nr Of Bytes to receive
static uint8_t  RxCnt;            // bytes received
static uint8_t  DataCnt;          // data words received
static uint8_t  TimeCnt;          // counter for auto display
static uint8_t  TimeoutCnt1;      // Timeout if no serial data received
static uint8_t  TimeoutCnt;       // Timeout if no serial data received
static char     *pt;

// Clear automatic register update array
static void ClearUpdates()
{
  for(uint8_t i=0;i<NROFNMEADATA;i++)
    RegisterNr[i]=0;
}

// Initialize NMEA Interface, must be called after InitUART()
void NMEAInit()
{
  DataCnt=0;
  NMEAFlags=0;
  ClearUpdates();
}

void NMEAEnable()
{
  RCIE=0;       // Receive Register will be polled by 20kHz SSP Interrupt to get better SSP Timing
  TXSTA=0x20;   // bit 2 BRGH=0;
  BAUDCON=0x00; // bit 3 BRG16=0
  SPBRG= 26-1;  // 26=9600 Baud @ 16 MHz BRG16=1 BRGH=1  baudrate= fosc/(64*(SPBRG+1))   SPRGB= 26 -1 = 9615 Baud
//SPBRG=104-1;  // 104=38400 Baud @ 16 MHz BRG16=0 BRGH=0  baudrate= fosc/(4*(SPBRG+1))   SPRGB= 104 -1 = 38461 Baud
  CREN=1;
  if(!ENABLEGPS)
  {
    uint8_t *pt=(uint8_t *)&NMEA;
    for(uint8_t i=0;i<sizeof(struct sNMEA);i++) pt[i]=0;
//  memset(&NMEA,0,sizeof(struct sNMEA)); // needs more code space
    ENABLEGPS=1;
  }
}

void NMEADisable()
{
  ENABLEGPS=0;
  InitUART(); // set Baudrate for PC Communication
}

// copy UTC into act_cl 6 byte as HHMMSS
static void NMEAGetUTCTime()
{
  for (uint8_t i = 0; i < 6; i++) // clock register is 6 digits HH:MM:SS
  		act_cl[5-i]=NMEA.UTCTime[i] & 0x0f; // copy GPS Time
}

// Show decimal Number from ASCII String
// buf    pointer to ASCII String with number digits and optional decimal, up to ten characters, nullterminated
// pos    position where to show number in display, 0 is leftmost digit
// degree 0= normal number, 1=latitude or longitude shown with degree character after digit 2
static void ShowNumber(char *buf, uint8_t pos,uint8_t degree)
{
char a,digits=11;

	if(degree)
  {
    act_reg[WSIZE-1-2]= 0;    // show 0o in case of empty string
    act_reg[WSIZE-1-3]= 0x0c; // show o as degree character in third digit
    digits--; // last displayed number should be 10 to have a blank digit before E or S
  }

  for(uint8_t i=0;pos<digits;i++,pos++) // maximum 10 digits
  {
    a=buf[i]; if(a==0) break; // null terminated

    if(a=='.') // decimal
    {
      pos--;
      act_reg[WSIZE-1-pos]|=0x80; // add decimal point to previous digit
    }
    else
    {
      act_reg[WSIZE-1-pos]= a & 0x0f; // expect number ASCII 0x30-0x39 -> 0-9
      if(degree && pos==2)  // skip degree digit at pos 3
         pos++;
    }
  }
}

// Store actual word into RAM register
// dataindex of data word to store 0-7 0= Time 1=Latitude, 2=Longitude etc.
// addr RAM register number from 0-7
static void NMEAStoreData(uint8_t dataindex,uint8_t addr)
{
uint8_t i,j,a,decimal;
char *buf;

  buf=(char *)&NMEA + NMEAptTable[dataindex]; // pointer to NMEA data, contains number with optional decimal in ASCII, convert to 56-bit BCD Format

  dest=act_reg; first=0;last=WSIZE-1; // destination is 14 bytes
  reg_zero();

  j=WSIZE-2;decimal=0;

  for(i=0;i<=10;i++) // max 10 digits
  {
    a=buf[i]; if(a==0) break;
    if(a=='.')
      decimal=i;
    else
      act_reg[j--]=a & 0x0f;
  }
  
// set exponent, use HH.MMSS format to be compatible with ->HMS ->H function
// NMEA Latitude 5711.123 will be converted to 5.711123 e01
// NMEA Longitude 00650123 E will be converted to -6.50123 e00 or 11950123 W -> 1.1950123 e02
// NMEA UTC 235959 will be converted to 2.35959000 e01  or 015959 -> 1.5959000 e00 

  if(dataindex==0 || dataindex==7) // Time doesn't have decimal because it is intentionally nullterminated after seconds, Satellites has 2 digits
    act_reg[0]=1;
  else if(dataindex<=2)   // latitude 0000.0000 decimal=4, longitude 00000.0000 decimal=5
    act_reg[0]=decimal-3; // latitude =  decimal=4decrement exponent until decimal encountered to place seconds behind comma
  else if(decimal)
    act_reg[0]=decimal-1; // exponent is decimal point

// Latitude and Longitude can be negative numbers if S or E

  if((dataindex==1 && NMEA.NS[0]=='S') || (dataindex==2 && NMEA.EW[0]=='E'))
    act_reg[WSIZE-1]=9;  // set minus sign, negative number

  reg_normalize(act_reg);
  dest=act_ram1[addr]; src=act_reg;
  reg_compress();
}

// Show actual NMEA Data in display depending on NMEAShow
// will be called every second when new data is received or when key pressed
// characters are valid for HP-25 and HP-21 hardware display drivers
// HP-29 hardware will show wrong characters 
void NMEAShowData()
{
uint8_t j;

  ClearInfo();

  if(!ENABLEGPS)  // !(NMEAFlags & NMEA_CONNECTED)) // !ENABLEGPS)
  {
      ReadText(16); // 16 = "GPS OFF" 27= "NO DATA"  
  }
  else
  if(NMEAShow == 0) // Show UTC ? parse buffer into register 
  {
    if(NMEA.Valid[0]>='1') // show "UTC" if Data is valid
    {
      ReadText(22); // UTC
    }
    else
    {
      for(uint8_t i=0;i<3;i++)
        if((NMEA.Satellites[1] & 0x0f)>i)  // Einer 
	      	act_reg[WSIZE-1-i]='O';  // o  show up to three signs for No Of Satellites
    }
		
    act_reg[WSIZE-1-6]='.';    // dot
    act_reg[WSIZE-1-9]='.';
    for(uint8_t i=0,j=4;i<6;i++,j++) // show 6 digits time, loop is compiled slightly shorter than without loop
    {
      act_reg[WSIZE-1-j]=NMEA.UTCTime[i] & 0x0f; // set HH.MM.SS
      if(i & 1) // skip one display digit after two number digits
        j++;
    }
  }
  else if(NMEAShow == 1) // show latitude ?
  {
    ShowNumber(NMEA.Latitude,1,1);
    if(NMEA.NS[0]!='N') // check for != to show S if nothing yet received
      act_reg[WSIZE-1-11]='S';             // show S if south
  }
  else if(NMEAShow == 2)  // show longitude ?
  {
    ShowNumber(NMEA.Longitude,0,1);
    if(NMEA.EW[0]!='W')  // check for != to show E if nothing yet received
      act_reg[WSIZE-1-11]='E';     // show E if East
  }
  else if(NMEAShow==3)  // Show Speed ?
  {
    ReadText(17); // SPEED
    if(NMEAFlags & NMEA_SHOWKNOTS)
      act_reg[WSIZE-1-5]='-';  // show minus if knots displayd
    ShowNumber(NMEA.Speed,6,0);
  }
  else if(NMEAShow==4)  // Show Track/Heading?
  {
    ReadText(18); // TRACK
    ShowNumber(NMEA.Track,6,0);
  }
  else if(NMEAShow==5) // Show Altitude ?
  {
    ReadText(19); // Altitude
    ShowNumber(NMEA.Altitude,6,0);
  }
  else if(NMEAShow==6) // Show HDOP ?
  {
    ReadText(20); // HDOP
    ShowNumber(NMEA.HDOP,6,0);
  }
  else if(NMEAShow==7) // Show Satellites ?
  {
    ReadText(21); // SAT
    ShowNumber(NMEA.Satellites,6,0);
  }
}

// NMEADo will be called every ms to read data from serial port 
// serial buffer will be full, 8 bytes 9600 Baud after 8.32 ms 
// return TRUE if data is received and should be displayed
//        FALSE data not yet received complete
uint8_t NMEADo()
{	
char a,i;

  while(CheckSer()) // NMEA Data available ?
  {
    a=ReadSer();  // read one character

    if(a=='$')    // $ character? new message comes in
    {
      DataCnt=0; // read message header GPGGA etc.
			goto L1;   // set receive buffer pointer to TagName and increment DataCnt
    }
    else if(a==',') // separator received ?
    {
      switch(DataCnt)
      {
        case 0: // skip data, wait for '$'
          break;

        case 1: // Message Header received GPGGA GPGTV etc.
          for(i=0;i<5;i++) // skip other messages than GPGGA GPVTG
              if(NMEA.TagName[i]!=GPGGA[i])
                 goto L2;
								 
          goto L1; // GPGGA received expect UTCTime DataCnt=1
L2:
          for(i=0;i<5;i++) // skip other messages than GPGGA GPVTG
              if(NMEA.TagName[i]!=GPVTG[i])
							  goto L3;

					DataCnt=10;  // GPVTG received, expect Track/Heading
          goto L1;
L3:
          DataCnt=0;    // stop GPGGA receiving, wait for $ 
          break;

        default:
          if(RxBytes)  // this implies pt!=NULL
            *pt=0;  // generate nullterminated string
          if(DataCnt==10) // altitude received ? stop GPGGA receive and wait for GPVTG data
            goto L3;
L1:
					pt=(char *)&NMEA; // NMEA+NMEApt[DataCnt];
          for(i=0;i<DataCnt;i++) // calculate offset of struct, this saves table space
          {
            RxBytes=RxBytesTab[i]; // number of bytes 
            pt+=RxBytes; // calculate pointer to buffer for received data
          }    
          RxBytes=RxBytesTab[DataCnt]; // set number of bytes to receive
					if(DataCnt==16) // Speed km/h ?
          {
            pt=(char *)&NMEA.Speed; // pointer has to be corrected, because buffer is used twice
            if(NMEAFlags & NMEA_SHOWKNOTS)
						  RxBytes=0;  // don't receive Speed km/h if knots is active, because they use the same buffer
          }
          RxCnt=0;

          if(++DataCnt>17)  // Last data received ?
					{
          	DataCnt=0;  // stop GPVTG receiving, wait for $
						
          	for(i=0;i<NROFNMEADATA;i++) // after data is received check for automatic register update
          	{
            	if(RegisterNr[i]) // value for this register specified ?
              	NMEAStoreData(RegisterNr[i]-1,i); // update register with value
          	}
            
          	if(++TimeCnt>=2)  // every 2 seconds change display in Auto Display mode
          	{
            	TimeCnt=0;
            	if(NMEAFlags & NMEA_AUTOPOS)
					      NMEAShow^=3; // show Latitude, Longitude alternating
            	if(NMEAFlags & NMEA_AUTOMENU)
					      NMEAShow=(NMEAShow+1) & (NROFNMEADATA-1); // select next menu
          	}

						TimeoutCnt=0; // retrigger Timeout
			    	NMEAFlags|=NMEA_CONNECTED;
          	return TRUE;
          }     
          break;
      }
    }
    else if(DataCnt && RxCnt+1<RxBytes) // receive incoming characters, reserve 1 byte for null termination
    {
        RxCnt++;
        *pt++=a;
    }
  }

  if(++TimeoutCnt1==0)
  {
    if(++TimeoutCnt==16)  // no data for >8*256 ms ? assume NMEA Module sends data every 1000 ms
    {
    	NMEAInit();  // reset NMEA_CONNECTED
      NMEADisable(); // disable GPS
    	return TRUE; // update display
    }
  }
  return FALSE;
}

// will be called if key pressed and NMEA display is active
void NMEADoKeys(uint8_t keyindex)
{
  TimeoutCnt=0; // allow GPS OFF Display
//  NMEAFlags|=NMEA_CONNECTED;

  NMEAFlags&=~NMEA_AUTOPOS;
  if(keyindex!=RSKEY)
    NMEAFlags&=~NMEA_AUTOMENU;      // any key stops Auto Display

  switch(keyindex)
  {
//    case -1:
//      ReadText(23);  // GPS ON
//      goto L1;
    case ENTERKEY:
      NMEAShow=(NMEAShow+1) & (NROFNMEADATA-1); // select next menu
      break;
    case RSKEY:
	    NMEAFlags^=NMEA_AUTOMENU;       // toggle Auto Display
      break;
    case CHSKEY:
      if(NMEAShow==3)
	    	NMEAFlags^=NMEA_SHOWKNOTS;      // toggle between knots and km/h Display
      break;
    case EEXKEY:
      if(!ENABLEGPS)
      {
        NMEAEnable();
        ReadText(23); // GPS ON
        goto L1;
      }
      else
      {
        NMEADisable();
//        ReadText(16); // GPS OFF
      }
      break;

    case ROLLKEY: // set RTC Time from GPS time
      if(NMEAFlags & NMEA_CONNECTED) // Zeit nur bernehmen, wenn Verbindung hergestellt
      {
        NMEAGetUTCTime();
#ifdef PCF2127
        RTCWriteDate();
#endif
      }
      break;
    case CLXKEY:
	    ClearUpdates();                 // stop all automatic updates
      break;
    case XYKEY:
      NMEAFlags|=NMEA_AUTOPOS;
			NMEAShow=1;
      break;
    default:
		  if(keyindex<NROFNMEADATA)              // number key 0-7, select menu, or store value into register
  		{
    		if(NMEAFlags & NMEA_STOKEY)     // STO prefix ?
    		{
      		if(NMEAFlags & NMEA_FKEY)     // f STO prefix ?
      		{
        		RegisterNr[keyindex]=NMEAShow+1; // Store value into register 0-7 repeatedly automatic update
      		} else
      		{
        		NMEAStoreData(NMEAShow,keyindex); // Store value into register 0-7 only once
        		RegisterNr[keyindex]=0;  // stop automatic update of this register
      		}
    		} else
      		NMEAShow=keyindex;            // number without prefix ? then select menu directly
  		}
      break; 
  }
  NMEAShowData();

L1:
// process prefix keys  f and STO
  NMEAFlags&=~NMEA_STOKEY;
	
  if(keyindex==STOKEY)              // STO key prefix
    NMEAFlags|=NMEA_STOKEY;
  else
  {
    NMEAFlags&=~NMEA_FKEY;          // don't reset FKEY if STO key is pressed
    if(keyindex==FKEY)              // f prefix
      NMEAFlags|=NMEA_FKEY;
  }
}
#endif
